﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddComplaintCommentCommandHandler(
           IComplaintCommentRepository _complaintCommentRepository,
           IComplaintRepository _complaintRepository,
           IUserRepository _userRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddComplaintCommentCommandHandler> _logger,
        UserInfoToken _userInfoToken) : IRequestHandler<AddComplaintCommentCommand, ServiceResponse<ComplaintCommentDto>>
    {
        public async Task<ServiceResponse<ComplaintCommentDto>> Handle(AddComplaintCommentCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var existComplaint = await _complaintRepository.All.Where(c => c.Id == request.ComplaintId).FirstOrDefaultAsync();
                if (existComplaint == null) 
                {
                    return ServiceResponse<ComplaintCommentDto>.Return404("Complaint not found");
                }
                var entity = _mapper.Map<ComplaintComment>(request);
                entity.CommentedAt = DateTime.UtcNow;
                entity.CommentedById =  _userInfoToken.Id;
                _complaintCommentRepository.Add(entity); 

                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<ComplaintCommentDto>.Return500();
                }
                var user = await _userRepository.All.Where(c => c.Id == entity.CommentedById).FirstOrDefaultAsync();
                if (user != null)
                {
                    entity.CommentedBy = user;
                }
                var entityDto = _mapper.Map<ComplaintCommentDto>(entity);
                return ServiceResponse<ComplaintCommentDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving Complaint Comment");
                return ServiceResponse<ComplaintCommentDto>.Return500("Error while saving Complaint Comment");
            }
        }
    }
}
