﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddComplaintInvestigationCommandHandler(
        IComplaintInvestigationRepository _scomplaintInvestigationRepository,
        IMapper _mapper,
        IUserRepository _userRepository,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddComplaintInvestigationCommandHandler> _logger,
         UserInfoToken _userInfoToken,
        ISendEmailRepository _sendEmailRepository,
        IUserNotificationRepository _userNotificationRepository,
        IComplaintRepository _complaintRepository) : IRequestHandler<AddComplaintInvestigationCommand, ServiceResponse<ComplaintInvestigationDto>>
    {
        public async Task<ServiceResponse<ComplaintInvestigationDto>> Handle(AddComplaintInvestigationCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entity = _mapper.Map<ComplaintInvestigation>(request);
                entity.InvestigationDate = DateTime.UtcNow;
                _scomplaintInvestigationRepository.Add(entity);
                var complaint=await _complaintRepository.All.Where(c=>c.Id==request.ComplaintId).FirstOrDefaultAsync();
                // Send Email Investigator User 
                try
                {
                    var userIds = new[] { _userInfoToken.Id, entity.InvestigatorId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var investigatorUser = users.FirstOrDefault(u => u.Id == entity.InvestigatorId);
                    if (investigatorUser != null)
                    {
                        var mainTitle = "Complaint Investigation Assignment Notification";
                        var subject = "Complaint Investigation Assignment";
                        var emailMessage = $"You have been assigned for  <b>{complaint.Title}</b> complaint as the Investigator for a Complaint Investigation. Please log in to the system to review the investigation details, conduct the investigation, record your findings, and provide recommendations before the follow-up date.";
                        _sendEmailRepository.AddAsignmentEmails(new SendEmail
                        {
                            Email = investigatorUser.Email,
                            FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                            ToName = investigatorUser.FirstName + ' ' + investigatorUser.LastName,
                            CreatedBy = fromUser.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, emailMessage, mainTitle, subject);

                        // Sent Notification
                        var notificationDto = new UserNotificationDto
                        {
                            Id = entity.Id,
                            UserId = entity.InvestigatorId,
                            NotificationsType = NotificationsType.COMPLAINT_INVESTIGATION,
                            Title = complaint != null ? complaint.Title : ""
                        };
                        _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                    }
                }
                catch (System.Exception ex)
                {
                    _logger.LogError(ex, "email does not sent");
                }
                if (await _uow.SaveAsync(cancellationToken) < 0)
                { 
                    return ServiceResponse<ComplaintInvestigationDto>.Return500();
                }
                var user = await _userRepository.All.Where(c => c.Id == entity.InvestigatorId).FirstOrDefaultAsync();
                if (user != null)
                {
                    entity.Investigator = user;
                }
                var entityDto = _mapper.Map<ComplaintInvestigationDto>(entity);
                return ServiceResponse<ComplaintInvestigationDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving ComplaintInvestigation");
                return ServiceResponse<ComplaintInvestigationDto>.Return500("Error while saving ComplaintInvestigation");
            }
        }
    }
}
