﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class UpdateComplaintTypeCommandHandler(
     IComplaintTypeRepository _complaintTypeRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IMapper _mapper,
    ILogger<UpdateComplaintTypeCommandHandler> logger) : IRequestHandler<UpdateComplaintTypeCommand, ServiceResponse<ComplaintTypeDto>>
{
    public async Task<ServiceResponse<ComplaintTypeDto>> Handle(UpdateComplaintTypeCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existingName = await _complaintTypeRepository.
                All.Where(c => c.Id != request.Id && c.Name.ToLower() == request.Name.ToLower()).FirstOrDefaultAsync();
            if (existingName != null)
            {
                return ServiceResponse<ComplaintTypeDto>.Return409("ComplaintType name is already exist");
            }
            var existingEntity = await _complaintTypeRepository.FindAsync(request.Id);
            if (existingEntity == null)
            {
                return ServiceResponse<ComplaintTypeDto>.Return404("ComplaintType does not exists.");
            }

            var entity = _mapper.Map(request, existingEntity);
            _complaintTypeRepository.Update(entity);
            if (await _uow.SaveAsync(cancellationToken) <= 0)
            {
                return ServiceResponse<ComplaintTypeDto>.Return500();
            }
            var entityDto = _mapper.Map<ComplaintTypeDto>(entity);
            return ServiceResponse<ComplaintTypeDto>.ReturnResultWith201(entityDto);
        }

        catch (Exception ex)
        {
            logger.LogError(ex, "Error while updating ComplaintType.");
            return ServiceResponse<ComplaintTypeDto>.Return500("Error while updating ComplaintType.");
        }
    }
}
