﻿using QualityManagement.Data;
using QualityManagement.Helper;
using QualityManagement.MediatR;
using QualityManagement.Repository;
using MediatR;
using Microsoft.EntityFrameworkCore;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class DownloadSessionMediaCommandHandler(
      ICourseSessionMediaRepository courseSessionMediaRepository,
       StorageServiceFactory _storeageServiceFactory,
       IStorageSettingRepository _storageSettingRepository,
       ICourseSessionMediaChunkRepository sessionMediaChunkRepository)
      : IRequestHandler<DownloadSessionMediaCommand, ServiceResponse<SessionMediaDownloadDto>>
{


    public async Task<ServiceResponse<SessionMediaDownloadDto>> Handle(DownloadSessionMediaCommand request, CancellationToken cancellationToken)
    {

        var courseSessionMedia = await courseSessionMediaRepository.All
            .FirstOrDefaultAsync(c => c.Id == request.SessionMediaId);

        if (courseSessionMedia == null)
        {
            return ServiceResponse<SessionMediaDownloadDto>.ReturnFailed(404, "Document  is not found");
        }

        if (courseSessionMedia.IsChunk)
        {
            var bytes = await CombineChunkBytes(courseSessionMedia);

            SessionMediaDownloadDto documentDownload = new SessionMediaDownloadDto
            {
                Data = bytes,
                ContentType = FileHelper.GetMimeTypeByExtension(courseSessionMedia.Extension),
                FileName = courseSessionMedia.Url,
            };
            return ServiceResponse<SessionMediaDownloadDto>.ReturnResultWith200(documentDownload);
        }
        else
        {

            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(courseSessionMedia.StorageSettingId);

            if (storeageSetting == null)
            {
                return ServiceResponse<SessionMediaDownloadDto>.ReturnFailed(404, "Storage setting not found");
            }

            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);

            var fileResult = await storageService.DownloadFileAsync(courseSessionMedia.Url, storeageSetting.JsonValue, courseSessionMedia.Key, courseSessionMedia.IV);

            if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
            {
                SessionMediaDownloadDto documentDownload = new SessionMediaDownloadDto
                {
                    Data = fileResult.FileBytes,
                    ContentType = FileHelper.GetMimeTypeByExtension(courseSessionMedia.Extension),
                    FileName = courseSessionMedia.Url,
                };
                return ServiceResponse<SessionMediaDownloadDto>.ReturnResultWith200(documentDownload);
            }

            return ServiceResponse<SessionMediaDownloadDto>.ReturnFailed(400, fileResult.ErrorMessage);
        }
    }
    private async Task<byte[]> CombineChunkBytes(CourseSessionMedia courseSessionMedia)
    {
        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(courseSessionMedia.StorageSettingId);

        if (storeageSetting == null)
        {
            return null;
        }

        var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);
        var documentChunks = await sessionMediaChunkRepository.All
            .Where(c => c.CourseSessionMediaId == courseSessionMedia.Id)
            .OrderBy(c => c.ChunkIndex).ToListAsync();
        var lstBytes = new List<byte[]>();
        foreach (var chunk in documentChunks)
        {
            var fileResult = await storageService.DownloadFileAsync(chunk.Url, storeageSetting.JsonValue, courseSessionMedia.Key, courseSessionMedia.IV);
            lstBytes.Add(fileResult.FileBytes);
        }
        using (var finalStream = new MemoryStream())
        {
            foreach (var chunk in lstBytes)
            {
                finalStream.Write(chunk, 0, chunk.Length);
            }
            return finalStream.ToArray();
        }
    }
}