﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR.CourseSessionMedias.DownloadSessionMedia
{
    public class GetCourseSessionMediaChunkCommandHandler(
    ICourseSessionMediaChunkRepository courseSessionMediaChunkRepository,
    ICourseSessionMediaRepository courseSessionMediaRepository,
     StorageServiceFactory _storeageServiceFactory,
     IStorageSettingRepository storageSettingRepository)
        : IRequestHandler<GetCourseSessionMediaChunkCommand, ServiceResponse<DocumentChunkDownload>>
    {
        public async Task<ServiceResponse<DocumentChunkDownload>> Handle(GetCourseSessionMediaChunkCommand request, CancellationToken cancellationToken)
        {
            var document = courseSessionMediaRepository.Find(request.CourseSessionMediaId);
            if (document == null)
            {
                return ServiceResponse<DocumentChunkDownload>.ReturnFailed(404, "Document is not found");
            }

            var documentChunk = await courseSessionMediaChunkRepository.All
                .Where(c => c.CourseSessionMediaId == request.CourseSessionMediaId && c.ChunkIndex == request.ChunkIndex)
                .FirstOrDefaultAsync();

            if (documentChunk == null)
            {
                return ServiceResponse<DocumentChunkDownload>.ReturnFailed(404, "Document Chunk is not found");
            }

            var storeageSetting = await storageSettingRepository.GetStorageSettingByIdOrLocal(document.StorageSettingId);

            if (storeageSetting == null)
            {
                return ServiceResponse<DocumentChunkDownload>.ReturnFailed(404, "Storage setting not found");
            }
            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);
            var fileResult = await storageService.DownloadFileAsync(documentChunk.Url, storeageSetting.JsonValue, document.Key, document.IV);
            if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
            {
                var extension = documentChunk.Extension;
                if (!extension.StartsWith("."))
                    extension = "." + extension;
                DocumentChunkDownload documentDownload = new DocumentChunkDownload
                {
                    Data = Convert.ToBase64String(fileResult.FileBytes),
                    ContentType = FileHelper.GetMimeTypeByExtension(extension),
                    FileName = documentChunk.Url,
                    ChunkIndex = documentChunk.ChunkIndex
                };
                return ServiceResponse<DocumentChunkDownload>.ReturnResultWith200(documentDownload);
            }
            return ServiceResponse<DocumentChunkDownload>.ReturnFailed(500, fileResult.ErrorMessage); ;
        }
    }
}
