﻿using AutoMapper;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.MediatR;
using QualityManagement.Repository;
using MediatR;
using Microsoft.Extensions.Logging;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class UploadSessionMediaChunkCommandHandler(
    ICourseSessionMediaRepository courseSessionMediaRepository,
    IStorageSettingRepository _storageSettingRepository,
    StorageServiceFactory _storeageServiceFactory,
    ICourseSessionMediaChunkRepository courseSessionMediaChunkRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IMapper _mapper,
    ILogger<UploadSessionMediaChunkCommandHandler> logger) : IRequestHandler<UploadSessionMediaChunkCommand, ServiceResponse<CourseSessionMediaChunkDto>>
{
    public async Task<ServiceResponse<CourseSessionMediaChunkDto>> Handle(UploadSessionMediaChunkCommand request, CancellationToken cancellationToken)
    {
        try
        {
            if (request.File == null)
            {
                return ServiceResponse<CourseSessionMediaChunkDto>.ReturnFailed(409, "Please select the file.");
            }


            var document = courseSessionMediaRepository.Find(request.CourseSessionMediaId);
            if (document == null)
            {
                return ServiceResponse<CourseSessionMediaChunkDto>.ReturnFailed(404, "Document not found.");
            }

            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(document.StorageSettingId);

            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);

            var fileNameKeyValut = await storageService.UploadFileChunkAsync(request.File, storeageSetting, document.Extension, document.Key, document.IV);

            if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
            {
                return ServiceResponse<CourseSessionMediaChunkDto>.Return422("Settings are not properly setup.");
            }
            var entity = new CourseSessionMediaChunk
            {
                Id = Guid.NewGuid(),
                ChunkIndex = request.ChunkIndex,
                CourseSessionMediaId = document.Id,
                Extension = request.Extension,
                Size = request.Size,
                Url = fileNameKeyValut.FileName,
                TotalChunk = request.TotalChunks
            };
            courseSessionMediaChunkRepository.Add(entity);

            if (await _uow.SaveAsync() <= 0)
            {
                return ServiceResponse<CourseSessionMediaChunkDto>.ReturnFailed(500, "Error While Added Document");
            }

            var entityDto = _mapper.Map<CourseSessionMediaChunkDto>(entity);
            entityDto.CourseSessionMediaId = document.Id;
            return ServiceResponse<CourseSessionMediaChunkDto>.ReturnResultWith200(entityDto);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error while storing media.");
            return ServiceResponse<CourseSessionMediaChunkDto>.Return500("Error while storing media");
        }
    }
}