﻿using AutoMapper;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using MediatR;
using Microsoft.Extensions.Logging;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;

namespace QualityManagement.MediatR;
public class UpdateCourseSessionCommandHandler(
    ICourseSessionRepository courseSessionRepository,
    IUnitOfWork<QMSDbContext> unitOfWork,
    ILogger<UpdateCourseSessionCommandHandler> logger,
    IMapper mapper)
    : IRequestHandler<UpdateCourseSessionCommand, ServiceResponse<CourseSessionDto>>
{
    public async Task<ServiceResponse<CourseSessionDto>> Handle(UpdateCourseSessionCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var course = await courseSessionRepository.FindAsync(request.Id);
            if (course == null)
            {
                return ServiceResponse<CourseSessionDto>.Return404("Course session does not exits");
            }

            var existingCourseSession = await courseSessionRepository.All
                .FirstOrDefaultAsync(d => d.CourseId == request.CourseId
                && d.Title.ToLower() == request.Title.ToLower() && d.Id != request.Id);

            if (existingCourseSession != null)
            {
                return ServiceResponse<CourseSessionDto>.Return409("Course session with the same title already exists.");
            }

            mapper.Map(request, course);

            courseSessionRepository.Update(course);
            if (await unitOfWork.SaveAsync() < 0)
            {
                return ServiceResponse<CourseSessionDto>.Return500("Error while saving course session");
            }

            return ServiceResponse<CourseSessionDto>.ReturnResultWith200(mapper.Map<CourseSessionDto>(course));
        }
        catch (System.Exception ex)
        {
            logger.LogError(ex, "Error while saving course session");
            return ServiceResponse<CourseSessionDto>.Return500("Error while saving course session");
        }
    }
}
