﻿using System;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class UpdateCourseCommandHandler(
    ICourseRepository courseRepository,
    IUnitOfWork<QMSDbContext> unitOfWork,
    ILogger<UpdateCourseCommandHandler> logger,
    IMapper mapper)
    : IRequestHandler<UpdateCourseCommand, ServiceResponse<CourseDto>>
{
    public async Task<ServiceResponse<CourseDto>> Handle(UpdateCourseCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var course = await courseRepository.FindAsync(request.Id);
            if (course == null)
            {
                return ServiceResponse<CourseDto>.Return404("Course not found.");
            }

            var existingCourse = await courseRepository.All
                .FirstOrDefaultAsync(c => c.Title.ToLower() == request.Title.ToLower()
                && c.Id != request.Id, cancellationToken);

            if (existingCourse != null)
            {
                return ServiceResponse<CourseDto>.Return409("Course with the same title already exists.");
            }

            course = mapper.Map(request, course);
            courseRepository.Update(course);

            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<CourseDto>.Return500("Error while updating course.");
            }

            return ServiceResponse<CourseDto>.ReturnResultWith201(mapper.Map<CourseDto>(course));

        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error while updating course.");
            return ServiceResponse<CourseDto>.Return500("Error while updating course.");
        }
    }
}
