﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateDepartmentCommandHandler(
        IDepartmentRepository _departmentRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UpdateDepartmentCommandHandler> _logger) : IRequestHandler<UpdateDepartmentCommand, ServiceResponse<DepartmentDto>>
    {
        public async Task<ServiceResponse<DepartmentDto>> Handle(UpdateDepartmentCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var existingName = await _departmentRepository.All
                    .Where(d => d.Id != request.Id && d.Name.ToLower() == request.Name.ToLower()).FirstOrDefaultAsync();
                if (existingName != null)
                {
                    return ServiceResponse<DepartmentDto>.Return409("Department name is already exist");
                }

                var existingEntity = await _departmentRepository.FindAsync(request.Id);
                if (existingEntity == null)
                {
                    return ServiceResponse<DepartmentDto>.Return404("Department Not Found");
                }

                var entity = _mapper.Map(request,existingEntity);
                _departmentRepository.Update(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<DepartmentDto>.Return500();
                }
                var entityDto = _mapper.Map<DepartmentDto>(entity);
                return ServiceResponse<DepartmentDto>.ReturnResultWith200(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while Updating  Department");
                return ServiceResponse<DepartmentDto>.Return500("Error while updating Department");
            }

        }
    }
}
