﻿using MediatR;
using Microsoft.AspNetCore.SignalR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class DocumentPermissionUserRoleCommandHandler(
    IDocumentRolePermissionRepository _documentRolePermissionRepository,
    IDocumentUserPermissionRepository _documentUserPermissionRepository,
    IUserNotificationRepository _userNotificationRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IDocumentAuditTrailRepository _documentAuditTrailRepository,
    UserInfoToken _userInfo,
    IUserRepository _userRepository,
    IDocumentRepository _documentRepository,
    ISendEmailRepository _sendEmailRepository,
    ICategoryRepository _categoryRepository,
    IHubContext<UserHub, IHubClient> _hubContext,
    IConnectionMappingRepository _connectionMappingRepository,
    ILogger<DocumentPermissionUserRoleCommandHandler> _logger) : IRequestHandler<DocumentPermissionUserRoleCommand, bool>
{
    public async Task<bool> Handle(DocumentPermissionUserRoleCommand request, CancellationToken cancellationToken)
    {
        List<DocumentAuditTrail> lstDocumentAuditTrail = new List<DocumentAuditTrail>();
        List<SendEmail> lstSendEmail = new List<SendEmail>();

        List<Guid> userIds = new List<Guid>();
        var currentUserInfo = _userRepository.Find(_userInfo.Id);

        if (request.Roles != null && request.Roles.Count() > 0)
        {
            List<DocumentRolePermission> lstDocumentRolePermission = new List<DocumentRolePermission>();

            foreach (var document in request.Documents)
            {
                foreach (var role in request.Roles)
                {

                    var existingPermission = _documentRolePermissionRepository.All.Where(c => c.DocumentId == Guid.Parse(document) && c.RoleId == Guid.Parse(role)).FirstOrDefault();
                    if (existingPermission != null)
                    {
                        _documentRolePermissionRepository.Remove(existingPermission);
                    }


                    lstDocumentRolePermission.Add(new DocumentRolePermission
                    {
                        DocumentId = Guid.Parse(document),
                        RoleId = Guid.Parse(role),
                        StartDate = request.StartDate,
                        EndDate = request.IsTimeBound ? request.EndDate.Value.AddDays(1).AddSeconds(-1) : request.EndDate,
                        IsTimeBound = request.IsTimeBound,
                        IsAllowDownload = request.IsAllowDownload,
                        CreatedBy = _userInfo.Id,
                        CreatedDate = DateTime.UtcNow
                    });

                    lstDocumentAuditTrail.Add(new DocumentAuditTrail()
                    {
                        DocumentId = Guid.Parse(document),
                        CreatedBy = _userInfo.Id,
                        CreatedDate = DateTime.UtcNow,
                        OperationName = DocumentOperation.Added_Permission,
                        AssignToRoleId = Guid.Parse(role)
                    });
                }
                List<Guid> roles = request.Roles.Select(c => Guid.Parse(c)).ToList();
                var documentInfo = await _documentRepository.FindAsync(Guid.Parse(document));
                documentInfo.IsShared = true;
                _documentRepository.Update(documentInfo);
                var users = await _userNotificationRepository.CreateRolesDocumentNotifiction(roles, documentInfo.Id);
                userIds.AddRange(users.Select(d => d.Id));
                if (request.IsAllowEmailNotification && users.Count() > 0)
                {


                    foreach (var user in users)
                    {
                        _sendEmailRepository.AddSharedEmails(new SendEmail
                        {
                            Email = user.Email,
                            FromEmail = currentUserInfo.Email,
                            FromName = currentUserInfo.FirstName + ' ' + currentUserInfo.LastName,
                            ToName = user.FirstName + ' ' + user.LastName,
                            CreatedBy = _userInfo.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, documentInfo.Name);
                    }

                }
            }
            _documentRolePermissionRepository.AddRange(lstDocumentRolePermission);
        }

        if (request.Users != null && request.Users.Count() > 0)
        {
            List<DocumentUserPermission> lstDocumentUserPermission = new List<DocumentUserPermission>();

            foreach (var document in request.Documents)
            {
                foreach (var user in request.Users)
                {

                    var existingPermission = _documentUserPermissionRepository.All.Where(c => c.DocumentId == Guid.Parse(document) && c.UserId == Guid.Parse(user)).FirstOrDefault();
                    if (existingPermission != null)
                    {
                        _documentUserPermissionRepository.Remove(existingPermission);
                    }


                    lstDocumentUserPermission.Add(new DocumentUserPermission
                    {
                        DocumentId = Guid.Parse(document),
                        UserId = Guid.Parse(user),
                        StartDate = request.StartDate,
                        EndDate = request.IsTimeBound ? request.EndDate.Value.AddDays(1).AddSeconds(-1) : request.EndDate,
                        IsTimeBound = request.IsTimeBound,
                        IsAllowDownload = request.IsAllowDownload,
                        CreatedBy = _userInfo.Id,
                        CreatedDate = DateTime.UtcNow
                    });

                    lstDocumentAuditTrail.Add(new DocumentAuditTrail()
                    {
                        DocumentId = Guid.Parse(document),
                        CreatedBy = _userInfo.Id,
                        CreatedDate = DateTime.UtcNow,
                        OperationName = DocumentOperation.Added_Permission,
                        AssignToUserId = Guid.Parse(user)
                    });

                }
                var filterUsers = request.Users.Where(c => !userIds.Contains(Guid.Parse(c))).ToList();
                var documentInfo = await _documentRepository.FindAsync(Guid.Parse(document));
                documentInfo.IsShared = true;
                _documentRepository.Update(documentInfo);
                if (request.IsAllowEmailNotification && filterUsers.Count() > 0)
                {
                    var users = await _userRepository.GetUsersByIds(filterUsers.Select(c => Guid.Parse(c)).ToList());
                    foreach (var user in users)
                    {
                        _sendEmailRepository.AddSharedEmails(new SendEmail
                        {
                            Email = user.Email,
                            FromEmail = currentUserInfo.Email,
                            FromName = currentUserInfo.FirstName + ' ' + currentUserInfo.LastName,
                            ToName = user.FirstName + ' ' + user.LastName,
                            CreatedBy = _userInfo.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, documentInfo.Name);
                    }

                }

                var tempUserIds = request.Users.Select(c => Guid.Parse(c)).ToList();
                _userNotificationRepository.CreateUsersDocumentNotifiction(tempUserIds, documentInfo.Id);
                userIds.AddRange(tempUserIds);
            }
            _documentUserPermissionRepository.AddRange(lstDocumentUserPermission);
        }
        if (lstSendEmail.Count() > 0)
        {
            _sendEmailRepository.AddRange(lstSendEmail);
        }

        if (lstDocumentAuditTrail.Count() > 0)
        {
            _documentAuditTrailRepository.AddRange(lstDocumentAuditTrail);
        }

        if (await _uow.SaveAsync() <= -1)
        {
            var errorDto = new DocumentRolePermissionDto
            {
                StatusCode = 500,
                Messages = new List<string> { "An unexpected fault happened. Try again later." }
            };
            return false;
        }
        foreach (var document in request.Documents)
        {
            var documentInfo = _documentRepository.Find(Guid.Parse(document));
            var category = _categoryRepository.All.Where(c => c.Id == documentInfo.CategoryId).FirstOrDefault();
            try
            {
                var onlineUsers = _connectionMappingRepository.GetAllUsersExceptThis(new SignlarUser { Id = _userInfo.Id.ToString() });
                if (onlineUsers.Count() > 0)
                {

                    var user = _connectionMappingRepository.GetUserInfoById(_userInfo.Id);
                    if (user != null)
                    {
                        await _hubContext.Clients.AllExcept(new List<string> { user.ConnectionId }).SendNotificationFolderChange(category.ParentId);
                    }
                    else
                    {
                        await _hubContext.Clients.All.SendNotificationFolderChange(category.ParentId);
                    }
                }
            }
            catch (Exception)
            {
                _logger.LogError("Error in sending notification to all users.");
            }
        }

        await _userNotificationRepository.SendNotification(userIds);

        return true;
    }
}
