﻿using AutoMapper;
using MediatR;
using Microsoft.AspNetCore.SignalR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddDocumentRolePermissionCommandHandler
     (IDocumentRolePermissionRepository _documentRolePermissionRepository,
      IUserNotificationRepository _userNotificationRepository,
      IUnitOfWork<QMSDbContext> _uow,
      IMapper _mapper,
      UserInfoToken _userInfo,
      IDocumentAuditTrailRepository _documentAuditTrailRepository,
      IUserRepository _userRepository,
      ISendEmailRepository _sendEmailRepository,
      ICategoryRepository _categoryRepository,
      IHubContext<UserHub, IHubClient> _hubContext,
      IConnectionMappingRepository _connectionMappingRepository,
      ILogger<AddDocumentRolePermissionCommandHandler> _logger,
      IDocumentRepository _documentRepository) : IRequestHandler<AddDocumentRolePermissionCommand, DocumentRolePermissionDto>
{
    public async Task<DocumentRolePermissionDto> Handle(AddDocumentRolePermissionCommand request, CancellationToken cancellationToken)
    {
        var permissions = _mapper.Map<List<DocumentRolePermission>>(request.DocumentRolePermissions);
        permissions.ForEach(permission =>
        {
            var existingPermission = _documentRolePermissionRepository.All.Where(c => c.DocumentId == permission.DocumentId && c.RoleId == permission.RoleId).FirstOrDefault();
            if (existingPermission != null)
            {
                _documentRolePermissionRepository.Remove(existingPermission);
            }

            if (permission.IsTimeBound)
            {
                permission.StartDate = permission.StartDate;
                permission.EndDate = permission.EndDate.Value.AddDays(1).AddSeconds(-1);
            }
        });
        _documentRolePermissionRepository.AddRange(permissions);
        var lstSendEmail = new List<SendEmail>();

        var documentId = request.DocumentRolePermissions.FirstOrDefault().DocumentId;
        var roleIds = request.DocumentRolePermissions.Select(c => c.RoleId).Distinct().ToList();
        List<DocumentAuditTrail> lstDocumentAuditTrail = new List<DocumentAuditTrail>();
        foreach (var roleId in roleIds)
        {
            var documentAudit = new DocumentAuditTrail()
            {
                DocumentId = documentId,
                CreatedBy = _userInfo.Id,
                CreatedDate = DateTime.UtcNow,
                OperationName = DocumentOperation.Added_Permission,
                AssignToRoleId = roleId
            };
            lstDocumentAuditTrail.Add(documentAudit);
        }
        if (lstDocumentAuditTrail.Count() > 0)
        {
            _documentAuditTrailRepository.AddRange(lstDocumentAuditTrail);
        }
        var documentInfo = await _documentRepository.FindAsync(documentId.Value);
        documentInfo.IsShared = true;
        _documentRepository.Update(documentInfo);
        var users = await _userNotificationRepository.CreateRolesDocumentNotifiction(roleIds, documentInfo.Id);
        var userIds = users.Select(c => c.Id).ToList();
        if (request.DocumentRolePermissions.Any(c => c.IsAllowEmailNotification == true) && userIds.Count() > 0)
        {

            var currentUserInfo = _userRepository.Find(_userInfo.Id);

            foreach (var user in users)
            {
                _sendEmailRepository.AddSharedEmails(new SendEmail
                {
                    Email = user.Email,
                    FromEmail = currentUserInfo.Email,
                    FromName = currentUserInfo.FirstName + ' ' + currentUserInfo.LastName,
                    ToName = user.FirstName + ' ' + user.LastName,
                    CreatedBy = _userInfo.Id,
                    CreatedDate = DateTime.UtcNow,
                }, documentInfo.Name);
            }

        }
        if (lstSendEmail.Count() > 0)
        {
            _sendEmailRepository.AddRange(lstSendEmail);
        }

        if (await _uow.SaveAsync() <= -1)
        {
            var errorDto = new DocumentRolePermissionDto
            {
                StatusCode = 500,
                Messages = new List<string> { "An unexpected fault happened. Try again later." }
            };
            return errorDto;
        }
        var category = _categoryRepository.All.Where(c => c.Id == documentInfo.CategoryId).FirstOrDefault();
        try
        {
            var onlineUsers = _connectionMappingRepository.GetAllUsersExceptThis(new SignlarUser { Id = _userInfo.Id.ToString() });
            if (onlineUsers.Count() > 0)
            {

                var userInfo = _connectionMappingRepository.GetUserInfoById(_userInfo.Id);
                if (userInfo != null)
                {
                    await _hubContext.Clients.AllExcept(new List<string> { userInfo.ConnectionId }).SendNotificationFolderChange(category.ParentId);
                }
                else
                {
                    await _hubContext.Clients.All.SendNotificationFolderChange(category.ParentId);
                }
            }
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "SignalR Error");
        }

        await _userNotificationRepository.SendNotification(userIds);
        //var documentid = request.DocumentRolePermissions.FirstOrDefault().DocumentId;
        //var roleIds = request.DocumentRolePermissions.Select(c => c.RoleId).Distinct().ToList();


        return new DocumentRolePermissionDto();
    }
}
