﻿using AutoMapper;
using MediatR;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class CreateDocumentShareableLinkCommandHandler
   (IMapper _mapper,
    IDocumentShareableLinkRepository _documentShareableLinkRepository,
    IDocumentRepository _documentRepository,
    IUnitOfWork<QMSDbContext> _uow) : IRequestHandler<CreateDocumentShareableLinkCommand, ServiceResponse<DocumentShareableLinkDto>>
{
    public async Task<ServiceResponse<DocumentShareableLinkDto>> Handle(CreateDocumentShareableLinkCommand request, CancellationToken cancellationToken)
    {
        DocumentShareableLink sharableLInk;
        if (request.Id.HasValue)
        {
            sharableLInk = _mapper.Map<DocumentShareableLink>(request);
            if (!string.IsNullOrWhiteSpace(sharableLInk.Password))
            {
                var plainTextBytes = System.Text.Encoding.UTF8.GetBytes(sharableLInk.Password);
                sharableLInk.Password = Convert.ToBase64String(plainTextBytes);
            }
            _documentShareableLinkRepository.Update(sharableLInk);
        }
        else
        {
            sharableLInk = _mapper.Map<DocumentShareableLink>(request);
            sharableLInk.LinkCode = Guid.NewGuid().ToString();
            if (!string.IsNullOrWhiteSpace(sharableLInk.Password))
            {
                var plainTextBytes = System.Text.Encoding.UTF8.GetBytes(sharableLInk.Password);
                sharableLInk.Password = Convert.ToBase64String(plainTextBytes);
            }
            _documentShareableLinkRepository.Add(sharableLInk);
        }

        if (await _uow.SaveAsync() <= 0)
        {
            return ServiceResponse<DocumentShareableLinkDto>.Return500();
        }
        var document = await _documentRepository.FindAsync(request.DocumentId);
        if (document == null)
        {
            return ServiceResponse<DocumentShareableLinkDto>.Return404("Document not found");
        }
        var result = _mapper.Map<DocumentShareableLinkDto>(sharableLInk);
        result.DocumentNumber = document.DocumentNumber;
        if (!string.IsNullOrWhiteSpace(result.Password))
        {
            var base64EncodedBytes = Convert.FromBase64String(result.Password);
            result.Password = System.Text.Encoding.UTF8.GetString(base64EncodedBytes);
        }
        return ServiceResponse<DocumentShareableLinkDto>.ReturnResultWith200(result);
    }
}
