﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class UploadNewDocumentChunkVersionCommandHandler(
     IDocumentRepository _documentRepository,
     IDocumentVersionRepository _documentVersionRepository,
     IUnitOfWork<QMSDbContext> _uow,
     IMapper _mapper,
     ILogger<UploadNewDocumentVersionCommandHandler> _logger,
     UserInfoToken _userInfoToken,
     IDocumentIndexRepository _documentIndexRepository,
     IDocumentAuditTrailRepository _documentAuditTrailRepository) : IRequestHandler<UploadNewDocumentChunkVersionCommand, ServiceResponse<DocumentVersionDto>>
{
    public async Task<ServiceResponse<DocumentVersionDto>> Handle(UploadNewDocumentChunkVersionCommand request, CancellationToken cancellationToken)
    {

        var doc = await _documentRepository.FindAsync(request.DocumentId);
        if (doc == null)
        {
            _logger.LogError("Document Not Found");
            return ServiceResponse<DocumentVersionDto>.Return500();
        }
        var currentDocumentVersion = await _documentVersionRepository.All.Where(c => c.DocumentId == request.DocumentId && c.IsCurrentVersion).FirstOrDefaultAsync();
        currentDocumentVersion.IsCurrentVersion = false;
        var url = Guid.NewGuid() + "." + request.Extension;

        var version = new DocumentVersion
        {
            DocumentId = doc.Id,
            Url = url,
            Key = doc.Key,
            IV = doc.IV,
            IsCurrentVersion = true,
            VersionNumber = await _documentVersionRepository.GetDocumentVersionCount(doc.Id) + 1,
            CreatedBy = _userInfoToken.Id,
            CreatedDate = DateTime.UtcNow,
            ModifiedBy = _userInfoToken.Id,
            ModifiedDate = DateTime.UtcNow,
            Comment = request.Comment,
            Extension = request.Extension,
            IsChunk = true,
            IsAllChunkUploaded = false,
            SignById = request.isSignatureExists ? _userInfoToken.Id : null,
            SignDate = request.isSignatureExists ? DateTime.UtcNow : null,
        };

        doc.ModifiedBy = _userInfoToken.Id;
        doc.ModifiedDate = DateTime.UtcNow;
        doc.IsSignatureExists = request.isSignatureExists;
        doc.IsChunk = true;
        doc.IsAllChunkUploaded = false;
        doc.SignById = request.isSignatureExists ? _userInfoToken.Id : null;
        doc.SignDate = request.isSignatureExists ? DateTime.UtcNow : null;
        doc.Url = url;
        doc.Extension = request.Extension;
        doc.Comment = request.Comment;

        var documentAudit = new DocumentAuditTrail()
        {
            DocumentId = doc.Id,
            CreatedBy = _userInfoToken.Id,
            CreatedDate = DateTime.UtcNow,
            OperationName = DocumentOperation.Added_Version,
            AssignToUserId = _userInfoToken.Id
        };

        _documentRepository.Update(doc);
        _documentVersionRepository.Update(currentDocumentVersion);
        _documentVersionRepository.Add(version);
        _documentIndexRepository.Add(new DocumentIndex { Id = Guid.NewGuid(), DocumentVersionId = version.Id });

        _documentAuditTrailRepository.Add(documentAudit);

        if (await _uow.SaveAsync() <= 0)
        {
            _logger.LogError("Error while adding industry");
            return ServiceResponse<DocumentVersionDto>.Return500();
        }
        var documentCommentDto = _mapper.Map<DocumentVersionDto>(version);
        return ServiceResponse<DocumentVersionDto>.ReturnResultWith200(documentCommentDto);
    }

}
