﻿using System;
using System.IO;
using System.Linq;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using J2N.Collections.Generic;
using MediatR;
using Microsoft.AspNetCore.Hosting;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class AddDocumentIndexContentCommandHandler(IDocumentIndexRepository _documentIndexRepository,
     IDocumentRepository _documentRepository,
     IDocumentVersionRepository documentVersionRepository,
     StorageServiceFactory _storeageServiceFactory,
     IStorageSettingRepository _storageSettingRepository,
     IWebHostEnvironment _webHostEnvironment,
     IDocumentChunkRepository documentChunkRepository,
      IUnitOfWork<QMSDbContext> _uow,
      ILogger<AddDocumentIndexContentCommandHandler> _logger,
      PathHelper pathHelper,
     IHttpClientFactory httpClientFactory,
    PathHelper _pathHelper) : IRequestHandler<AddDocumentIndexContentCommand, bool>
{
    public async Task<bool> Handle(AddDocumentIndexContentCommand request, CancellationToken cancellationToken)
    {
        var documentIndex = await _documentIndexRepository.All.OrderByDescending(c => c.CreatedDate).FirstOrDefaultAsync();
        if (documentIndex == null)
        {
            return false;
        }
        var documentVersion = await documentVersionRepository.All.FirstOrDefaultAsync(c => c.Id == documentIndex.DocumentVersionId);
        if (documentVersion == null && !documentVersion.IsAllChunkUploaded)
        {
            return false;
        }

        var document = await _documentRepository.All.FirstOrDefaultAsync(c => c.Id == documentVersion.DocumentId);
        if (document != null)
        {
            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(document.StorageSettingId);
            if (storeageSetting == null)
            {
                return false;
            }
            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);
            byte[] responseBytes;
            if (documentVersion.IsChunk)
            {
                responseBytes = await CombineChunkBytes(documentVersion, document, storeageSetting, storageService);
            }
            else
            {
                var fileResult = await storageService.DownloadFileAsync(documentVersion.Url, storeageSetting.JsonValue, document.Key, document.IV);
                responseBytes = fileResult.FileBytes;
            }
            try
            {
                string extension = Path.GetExtension(document.Url);
                var extractor = ContentExtractorFactory.GetExtractor(extension, _webHostEnvironment.WebRootPath);
                var imagessupport = _pathHelper.IMAGESSUPPORT;
                var tessLang = _pathHelper.TESSSUPPORTLANGUAGES;
                if (extractor != null)
                {
                    string tessdataPath = Path.Combine(_webHostEnvironment.WebRootPath, _pathHelper.TESSDATA);
                    var content = new DocumentProcessor(extractor).ProcessDocumentByBytes(responseBytes, tessdataPath, _pathHelper.TESSSUPPORTLANGUAGES);
                    content = UnWantKeywordRemovalHelper.CleanExtractedText(content);
                    if (!string.IsNullOrEmpty(content))
                    {
                        //await extractSummaryOfDocument(content);
                        string searchIndexPath = Path.Combine(_webHostEnvironment.WebRootPath, _pathHelper.SearchIndexPath);
                        var indexWriterManager = new IndexWriterManager(searchIndexPath);
                        indexWriterManager.AddDocument(documentVersion.Id.ToString(), content);
                        indexWriterManager.Commit();
                        indexWriterManager.Dispose();
                    }
                }
                else if (Array.Exists(imagessupport, element => element.ToLower() == extension.ToLower()))
                {
                    ExtractTEssData(responseBytes, documentVersion.Id, tessLang);
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "Error while indexing document");
            }

        }
        _documentIndexRepository.Remove(documentIndex);
        if (await _uow.SaveAsync() <= 0)
        {
            _logger.LogError("Error while saving data");
        }
        return true;
    }

    private async Task<byte[]> CombineChunkBytes(DocumentVersion documentVersion, Document document, StorageSetting storeageSetting, IStorageService storageService)
    {
        var documentChunks = await documentChunkRepository.All.Where(c => c.DocumentVersionId == documentVersion.Id).OrderBy(c => c.ChunkIndex).ToListAsync();
        var lstBytes = new List<byte[]>();
        foreach (var chunk in documentChunks)
        {
            var fileResult = await storageService.DownloadFileAsync(chunk.Url, storeageSetting.JsonValue, document.Key, document.IV);
            lstBytes.Add(fileResult.FileBytes);
        }
        using (var finalStream = new MemoryStream())
        {
            foreach (var chunk in lstBytes)
            {
                finalStream.Write(chunk, 0, chunk.Length);
            }
            return finalStream.ToArray();
        }
    }

    private void ExtractTEssData(byte[] fileBytes, Guid id, string tessLang)
    {
        string tessFilePath = Path.Combine(_webHostEnvironment.WebRootPath, _pathHelper.TESSDATA);
        var tessDataContextExtractor = new TessDataContextExtractor();
        var content = tessDataContextExtractor.ExtractContentByBytes(tessFilePath, fileBytes, tessLang, _logger);
        content = UnWantKeywordRemovalHelper.CleanExtractedText(content);
        if (!string.IsNullOrEmpty(content))
        {
            string searchIndexPath = Path.Combine(_webHostEnvironment.WebRootPath, _pathHelper.SearchIndexPath);
            var indexWriterManager = new IndexWriterManager(searchIndexPath);
            indexWriterManager.AddDocument(id.ToString(), content);
            indexWriterManager.Commit();
            indexWriterManager.Dispose();
        }
    }
}
