﻿using Amazon.Runtime.Internal.Util;
using MediatR;
using Microsoft.Extensions.Logging;
using Mscc.GenerativeAI;
using QualityManagement.Data.Entities;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class FileDownloadRequestCommandHandler(
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storeageServiceFactory,
        ILogger<FileDownloadRequestCommandHandler> _logger) : IRequestHandler<FileDownloadRequestCommand, ServiceResponse<DocumentDownload>>
    {
        public async Task<ServiceResponse<DocumentDownload>> Handle(FileDownloadRequestCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var fileBytes = await DownloadChunkBytes(request);
                DocumentDownload DocumentDownload = new DocumentDownload
                {
                    Data = fileBytes,
                    ContentType = FileHelper.GetMimeTypeByExtension(request.Extension),
                    FileName = request.FileName
                };
                return ServiceResponse<DocumentDownload>.ReturnResultWith200(DocumentDownload);
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "error while getting Office PreView File Download");
                return ServiceResponse<DocumentDownload>.Return500("An unexpected fault happened. Try again later. ");
            }

        }
        private async Task<byte[]> DownloadChunkBytes(FileDownloadRequestCommand request)
        {
            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(request.StorageSettingId);

            if (storeageSetting == null)
            {
                return null;
            }
            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);
            var lstBytes = new List<byte[]>();
            var orderedChunks = request.Chunks.OrderBy(c => c.ChunkIndex).ToList();
            foreach (var chunk in orderedChunks)
            {
                var fileResult = await storageService.DownloadFileAsync(chunk.Url, storeageSetting.JsonValue, request.Key, request.IV);
                lstBytes.Add(fileResult.FileBytes);
            }
            using (var finalStream = new MemoryStream())
            {
                foreach (var chunk in lstBytes)
                {
                    finalStream.Write(chunk, 0, chunk.Length);
                }
                return finalStream.ToArray();
            }
        }
    }
}
