﻿using AutoMapper;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddEmailTemplateCommandHandler(
    IEmailTemplateRepository emailTemplateRepository,
    IMapper mapper,
    IUnitOfWork<QMSDbContext> uow,
    UserInfoToken userInfoToken,
    ILogger<AddEmailTemplateCommandHandler> logger
    ) : IRequestHandler<AddEmailTemplateCommand, ServiceResponse<EmailTemplateDto>>
{
    public async Task<ServiceResponse<EmailTemplateDto>> Handle(AddEmailTemplateCommand request, CancellationToken cancellationToken)
    {
        var entityExist = await emailTemplateRepository.FindBy(c => c.Name == request.Name)
            .FirstOrDefaultAsync(cancellationToken);

        if (entityExist != null)
        {
            logger.LogError("Email Template already exist.");
            return ServiceResponse<EmailTemplateDto>.Return409("Email Template already exist.");
        }

        var entity = mapper.Map<EmailTemplate>(request);
        entity.Id = Guid.NewGuid();
        entity.CreatedBy = userInfoToken.Id;
        entity.CreatedDate = DateTime.UtcNow;
        entity.ModifiedBy = userInfoToken.Id;
        emailTemplateRepository.Add(entity);

        if (await uow.SaveAsync(cancellationToken) <= 0)
        {
            return ServiceResponse<EmailTemplateDto>.Return500();
        }
        var entityDto = mapper.Map<EmailTemplateDto>(entity);
        return ServiceResponse<EmailTemplateDto>.ReturnResultWith200(entityDto);
    }
}
