﻿using System;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class CreateEmployeeCourseSessionCommandHandler(
    IEmployeeCourseSessionRepository employeeCourseSessionRepository,
    IUnitOfWork<QMSDbContext> unitOfWork,
    ILogger<CreateEmployeeCourseSessionCommandHandler> logger,
    IMapper mapper,
    ICourseSessionQuizRepository courseSessionQuizRepository)
    : IRequestHandler<CreateEmployeeCourseSessionCommand, ServiceResponse<EmployeeCourseSessionDto>>
{
    public async Task<ServiceResponse<EmployeeCourseSessionDto>> Handle(CreateEmployeeCourseSessionCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existingSession = await employeeCourseSessionRepository.All.FirstOrDefaultAsync(x =>
                x.EmployeeId == request.EmployeeId && x.CourseSessionId == request.CourseSessionId);

            if (existingSession != null)
            {
                //return ServiceResponse<EmployeeCourseSessionDto>.Return409("Session already assign to employee.");
                return ServiceResponse<EmployeeCourseSessionDto>
                    .ReturnResultWith200(mapper.Map<EmployeeCourseSessionDto>(existingSession));
            }
            var employeeCourseSession = new EmployeeCourseSession
            {
                EmployeeId = request.EmployeeId,
                CourseSessionId = request.CourseSessionId
            };
            employeeCourseSessionRepository.Add(employeeCourseSession);

            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<EmployeeCourseSessionDto>.Return500("Failed to assign session to employee.");
            }

            var employeeCourseSessionDto = new EmployeeCourseSessionDto
            {
                Id = employeeCourseSession.Id,
                EmployeeId = employeeCourseSession.EmployeeId,
                CourseSessionId = employeeCourseSession.CourseSessionId,
            };

            return ServiceResponse<EmployeeCourseSessionDto>.ReturnResultWith200(employeeCourseSessionDto);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error occurred while creating employee course session.");
            return ServiceResponse<EmployeeCourseSessionDto>.Return500("An error occurred while creating employee course session.");
        }
    }
}
