﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data;
using QualityManagement.Helper;
using QualityManagement.MediatR;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

public class GetEmployeeCourseSessionMediaStatusCommandHandler(
    ICourseSessionMediaRepository courseSessionMediaRepository,
    IEmployeeCourseSessionRepository employeeCourseSessionRepository,
    IEmployeeCourseSessionMediaRepository employeeCourseSessionMediaRepository,
    ILogger<GetEmployeeCourseSessionMediaStatusCommandHandler> logger)
    : IRequestHandler<GetEmployeeCourseSessionMediaStatusCommand, ServiceResponse<List<EmployeeCourseSessionMediaCompleteDto>>>
{
    public async Task<ServiceResponse<List<EmployeeCourseSessionMediaCompleteDto>>> Handle(GetEmployeeCourseSessionMediaStatusCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var allMedia = await courseSessionMediaRepository.All
                .Where(c => c.CourseSessionId == request.CourseSessionId)
                .Include(c => c.CourseSession)
                .ToListAsync(cancellationToken);

            if (!allMedia.Any())
                return ServiceResponse<List<EmployeeCourseSessionMediaCompleteDto>>.Return404("No media found for this session");

            var employeeCourseSession = await employeeCourseSessionRepository.All
                .Where(x => x.EmployeeCourse.Link == request.CourseLink
                        && x.CourseSessionId == request.CourseSessionId)
                .OrderByDescending(x => x.CompletedDate)
                .FirstOrDefaultAsync(cancellationToken);

            var watchedIds = new List<Guid>();

            if (employeeCourseSession != null)
            {
                watchedIds = (await employeeCourseSessionMediaRepository.All
                    .Where(x => x.EmployeeCourseSessionId == employeeCourseSession.Id)
                    .Select(x => x.CourseSessionMediaId)
                    .ToListAsync(cancellationToken)).ToList();
            }

            var mediaStatusList = allMedia.Select(m => new EmployeeCourseSessionMediaCompleteDto
            {
                Id = m.Id,
                CourseSessionId = m.CourseSessionId,
                Url = m.Url,
                Name = m.Name,
                Extension = m.Extension,
                IsChunk = m.IsChunk,
                MediaType = m.MediaType,
                IsViewed = watchedIds.Contains(m.Id)
            }).ToList();

            return ServiceResponse<List<EmployeeCourseSessionMediaCompleteDto>>.ReturnResultWith200(mediaStatusList);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error while getting course session media status");
            return ServiceResponse<List<EmployeeCourseSessionMediaCompleteDto>>.Return500("An error occurred while processing your request.");
        }
    }
}
