﻿using Amazon.Runtime.Internal.Util;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.MediatR;
using QualityManagement.Repository;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

public class GetResumeCourseSessionMediaHandler(
    ICourseSessionRepository courseSessionRepository,
    IEmployeeCourseRepository employeeCourseRepository,
    IEmployeeCourseSessionMediaRepository employeeCourseSessionMediaRepository,
    ILogger<GetResumeCourseSessionMediaHandler> _logger
) : IRequestHandler<GetResumeCourseSessionMedia, ServiceResponse<CourseSessionMediaDataDto>>
{
    public async Task<ServiceResponse<CourseSessionMediaDataDto>> Handle(GetResumeCourseSessionMedia request, CancellationToken cancellationToken)
    {
        try
        {
            var employeeCourse = await employeeCourseRepository.All
           .AsNoTracking()
           .FirstOrDefaultAsync(c => c.Link == request.Link, cancellationToken);

            if (employeeCourse == null)
                return ServiceResponse<CourseSessionMediaDataDto>.Return404("EmployeeCourse not found");

            var courseSessions = await courseSessionRepository.All
                .Where(cs => cs.CourseId == employeeCourse.CourseId)
                .Include(cs => cs.CourseSessionMedias)
                .OrderBy(cs => cs.CreatedDate)
                .ToListAsync(cancellationToken);

            if (!courseSessions.Any())
                return ServiceResponse<CourseSessionMediaDataDto>.Return404("No course sessions found");

            var sessionIds = courseSessions.Select(cs => cs.Id).ToList();

            var lastWatched = await employeeCourseSessionMediaRepository.All
                .Where(e => sessionIds.Contains(e.CourseSessionMedia.CourseSessionId))
                .OrderByDescending(e => e.ModifiedDate)
                .Select(e => new
                {
                    e.CourseSessionMediaId,
                    CourseSessionId = e.CourseSessionMedia.CourseSessionId
                })
                .FirstOrDefaultAsync(cancellationToken);

            if (lastWatched != null)
            {
                var currentSession = courseSessions.FirstOrDefault(s => s.Id == lastWatched.CourseSessionId);

                if (currentSession != null)
                {
                    var mediaList = currentSession.CourseSessionMedias
                        .OrderBy(m => m.CreatedDate)
                        .ToList();

                    int currentMediaIndex = mediaList.FindIndex(m => m.Id == lastWatched.CourseSessionMediaId);

                    if (currentMediaIndex + 1 < mediaList.Count)
                    {
                        return ServiceResponse<CourseSessionMediaDataDto>.ReturnResultWith200(new CourseSessionMediaDataDto
                        {
                            CourseSessionId = currentSession.Id,
                            CourseSessionMediaId = mediaList[currentMediaIndex + 1].Id
                        });
                    }

                    int currentSessionIndex = courseSessions.FindIndex(s => s.Id == currentSession.Id);
                    if (currentSessionIndex + 1 < courseSessions.Count)
                    {
                        var nextSession = courseSessions[currentSessionIndex + 1];
                        var firstMedia = nextSession.CourseSessionMedias
                            .OrderBy(m => m.CreatedDate)
                            .FirstOrDefault();

                        if (firstMedia != null)
                        {
                            return ServiceResponse<CourseSessionMediaDataDto>.ReturnResultWith200(new CourseSessionMediaDataDto
                            {
                                CourseSessionId = nextSession.Id,
                                CourseSessionMediaId = firstMedia.Id
                            });
                        }
                    }
                    return ServiceResponse<CourseSessionMediaDataDto>.ReturnResultWith200(new CourseSessionMediaDataDto
                    {
                        CourseSessionId = currentSession.Id,
                        CourseSessionMediaId = mediaList[currentMediaIndex].Id
                    });
                }
            }

            var firstSession = courseSessions.First();
            var firstMediaInFirstSession = firstSession.CourseSessionMedias
                .OrderBy(m => m.CreatedDate)
                .FirstOrDefault();

            if (firstMediaInFirstSession == null)
                return ServiceResponse<CourseSessionMediaDataDto>.Return404("No media found");

            return ServiceResponse<CourseSessionMediaDataDto>.ReturnResultWith200(new CourseSessionMediaDataDto
            {
                CourseSessionId = firstSession.Id,
                CourseSessionMediaId = firstMediaInFirstSession.Id
            });
        }
        catch(System.Exception ex)
        {
            _logger.LogError(ex, "Error while Get Resume Course Session and Media ");
            return ServiceResponse<CourseSessionMediaDataDto>.Return500("Error while Get Resume Course Session and Media");
        }
       
    }
}
