﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.EmployeeCourses;

public class AssignCourseToEmployeeCommandHandler(
IEmployeeCourseRepository employeeCourseRepository,
IUserRepository userRepository,
IUnitOfWork<QMSDbContext> unitOfWork,
ILogger<AssignCourseToEmployeeCommandHandler> logger,
ICourseRepository courseRepository,
 ICourseSessionRepository _courseSessionRepository,
 IUserNotificationRepository _userNotificationRepository)
: IRequestHandler<AssignCourseToEmployeeCommand, ServiceResponse<bool>>
{
    public async Task<ServiceResponse<bool>> Handle(AssignCourseToEmployeeCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var course = await courseRepository.FindAsync(request.CourseId);
            if (course == null)
            {
                return ServiceResponse<bool>.Return409("Course not exists.");
            }
            var hasCourseSessions = await _courseSessionRepository.All
                .AnyAsync(c => c.CourseId == course.Id);

            if (!hasCourseSessions)
            {
                return ServiceResponse<bool>.Return409("Course does not have any sessions.");
            }
            var existingEmployeeIds = await employeeCourseRepository.All
                .Where(d => d.CourseId == request.CourseId)
                .Select(d => d.EmployeeId)
                .ToListAsync(cancellationToken);

            var employeeCourses = new List<EmployeeCourse>();

            if (request.IsAssignToAllEmployee)
            {
                var employeeIds = await userRepository.All
                    .Where(d => !existingEmployeeIds.Contains(d.Id))
                    .Select(e => e.Id).ToListAsync();

                foreach (var employeeId in employeeIds)
                {
                    employeeCourses.Add(new EmployeeCourse
                    {
                        EmployeeId = employeeId,
                        CourseId = request.CourseId,
                        Status = EmployeeCourseStatus.Pending,
                        EmailStatus = EmailStatus.Pending,
                        //TODO: change it from UI.
                        //DueDate = DateTime.UtcNow,
                        Link = GeneratCourceLink.GenerateLink(Guid.NewGuid().ToString())
                    });
                    _userNotificationRepository.AddUserAsignedNewCourseNotification(request.CourseId, employeeId, course.Title);
                }
            }
            else
            {
                var employeeIds = request.EmployeeIds.Where(d => !existingEmployeeIds.Contains(d)).ToList();
                foreach (var employeeId in employeeIds)
                {
                    employeeCourses.Add(new EmployeeCourse
                    {
                        EmployeeId = employeeId,
                        CourseId = request.CourseId,
                        Status = EmployeeCourseStatus.Pending,
                        EmailStatus = EmailStatus.Pending,
                        //TODO: change it from UI.
                        //DueDate = DateTime.UtcNow,
                        Link = GeneratCourceLink.GenerateLink(Guid.NewGuid().ToString())
                    });
                    _userNotificationRepository.AddUserAsignedNewCourseNotification(request.CourseId, employeeId, course.Title);
                }
            }
            employeeCourseRepository.AddRange(employeeCourses);


            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<bool>.Return500("Error while assigning course to company");
            }

            return ServiceResponse<bool>.ReturnSuccess();
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error while assigning course to company");
            return ServiceResponse<bool>.Return500("Error while assigning course to company");
        }
    }

    //private static string GenerateLink(string plainText)
    //{
    //    // 16 bytes = 128 bits of entropy (like a GUID)
    //    var randomBytes = new byte[32];
    //    using (var rng = RandomNumberGenerator.Create())
    //    {
    //        rng.GetBytes(randomBytes);
    //    }

    //    // Base64 encode and make URL-safe
    //    var token = Convert.ToBase64String(randomBytes)
    //        .Replace("+", "-")
    //        .Replace("/", "_")
    //        .Replace("=", ""); // remove padding

    //    return token;
    //}
}