﻿using MediatR;
using Microsoft.AspNetCore.Hosting;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto.Course;
using QualityManagement.Data.Entities;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.IO;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class DownloadEmployeeCourseCertificateCommandHandler(
    IEmployeeCourseRepository employeeCourseRepository,
    EmployeeCertificateGenerator employeeCertificateGenerator,
    ILogger<DownloadEmployeeCourseCertificateCommandHandler> logger)
    : IRequestHandler<DownloadEmployeeCourseCertificateCommand, ServiceResponse<CertificateDto>>
{
    public async Task<ServiceResponse<CertificateDto>> Handle(DownloadEmployeeCourseCertificateCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var employeeCourse = await employeeCourseRepository
                .All
                .Include(d => d.Course)
                .Include(d => d.Employee)
                .FirstOrDefaultAsync(ec => ec.Id == request.Id);
            if (employeeCourse == null)
            {
                return ServiceResponse<CertificateDto>.ReturnFailed(404, "Employee course not found");
            }

            if (employeeCourse.Status != EmployeeCourseStatus.Completed)
            {
                return ServiceResponse<CertificateDto>.ReturnFailed(400, "Certificate is not available for this course");
            }

            var bytes =await  employeeCertificateGenerator.GenerateCertificate(
                    employeeCourse.Id,
                    employeeCourse.Course.Title,
                    $"{employeeCourse.Employee.FirstName} {employeeCourse.Employee.LastName}",
                    employeeCourse.CompletedDate.Value.Date);
            if (bytes == null || bytes.Length == 0)
            {
                return ServiceResponse<CertificateDto>.ReturnFailed(500, "Error generating certificate");
            }
            var certificate = new CertificateDto
            {
                Data = bytes,
                ContentType = "application/pdf",
                FileName = $"{employeeCourse.Id}.pdf"
            };
            return ServiceResponse<CertificateDto>.ReturnResultWith200(certificate);

        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error generating certificate");
            return ServiceResponse<CertificateDto>.ReturnFailed(500, "Error generating certificate");
        }
    }
}
