﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class GetEmployeeCourseByLinkCommandHandler(
    IEmployeeCourseRepository employeeCourseRepository,
    ILogger<GetEmployeeCourseSessionByLinkCommandHandler> logger
    ) : IRequestHandler<GetEmployeeCourseByLinkCommand, ServiceResponse<EmployeeCourseDetailDto>>
{
    public async Task<ServiceResponse<EmployeeCourseDetailDto>> Handle(GetEmployeeCourseByLinkCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var employeeCourse = await employeeCourseRepository
                .All
                .Include(d => d.Employee)
                .Include(d => d.Course)
                    .ThenInclude(d => d.CourseSessions)
                        .ThenInclude(d => d.CourseSessionQuiz)
                .Include(d => d.EmployeeCourseSessions)
                .Where(d => d.Link == request.Link)
                .FirstOrDefaultAsync();

            var employeeCourseDto = new EmployeeCourseDetailDto
            {
                Id = employeeCourse.Id,
                CourseId = employeeCourse.CourseId,
                CourseName = employeeCourse.Course.Title,
                Status = employeeCourse.Status,
                CourseDescription = employeeCourse.Course.Description,
                CompletedDate = employeeCourse.CompletedDate,
                EmployeeName = $"{employeeCourse.Employee.FirstName} {employeeCourse.Employee.LastName}",
                CourseSessions = employeeCourse.Course.CourseSessions.Select(session => new CourseSessionDto
                {
                    Id = session.Id,
                    Title = session.Title,
                    Description = session.Description,
                    CourseId = session.CourseId,
                    CourseName = session.Course.Title,
                    HasQuiz = session.CourseSessionQuiz != null
                }).ToList(),
            };


            if (employeeCourse == null)
            {
                return ServiceResponse<EmployeeCourseDetailDto>.Return404("Employee Course not found.");
            }
            return ServiceResponse<EmployeeCourseDetailDto>.ReturnResultWith200(employeeCourseDto);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error occurred while getting employee course session.");
            return ServiceResponse<EmployeeCourseDetailDto>.Return500("An error occurred while getting employee course session.");
        }
    }
}
