﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.EmployeeCoursesSessionMedias;

public class MarkSessionMediaAsReadCommandHandler(
    ILogger<MarkSessionMediaAsReadCommandHandler> logger,
    ICourseSessionMediaRepository courseSessionMediaRepository,
    IEmployeeCourseRepository employeeCourseRepository,
    IEmployeeCourseSessionRepository employeeCourseSessionRepository,
    ICourseSessionQuizRepository courseSessionQuizRepository,
    IEmployeeCourseSessionMediaRepository employeeCourseSessionMediaRepository,
    IUnitOfWork<QMSDbContext> unitOfWork)
    : IRequestHandler<MarkSessionMediaAsReadCommand, ServiceResponse<bool>>
{
    public async Task<ServiceResponse<bool>> Handle(MarkSessionMediaAsReadCommand request, CancellationToken cancellationToken)
    {
        var media = await courseSessionMediaRepository.FindAsync(request.MediaId);
        if (media == null)
        {
            logger.LogError($"Media does not exists with id {request.MediaId}");
            return ServiceResponse<bool>.ReturnSuccess();
        }

        var employeeCourse = await employeeCourseRepository.FindAsync(request.EmployeeCourseId);
        if (employeeCourse == null)
        {
            logger.LogError("Employee course does not exists.");
            return ServiceResponse<bool>.ReturnSuccess();
        }

        var allSessionMedias = await courseSessionMediaRepository.All
            .Where(d => d.CourseSessionId == media.CourseSessionId)
            .ToListAsync(cancellationToken);

        var employeeCourseSession = await employeeCourseSessionRepository.All
            .FirstOrDefaultAsync(d => d.EmployeeCourseId == request.EmployeeCourseId
                        && d.CourseSessionId == media.CourseSessionId);

       bool isNewEmployeeCourseSession = false;
        if (employeeCourseSession == null)
        {
            var employeeCourseSessionId = Guid.NewGuid();
            employeeCourseSession = new EmployeeCourseSession
            {
                Id = employeeCourseSessionId,
                CourseSessionId = media.CourseSessionId,
                EmployeeCourseId = request.EmployeeCourseId,
                EmployeeId = employeeCourse.EmployeeId,
                EmployeeCourseSessionMedias = new List<EmployeeCourseSessionMedia>
                {
                    new EmployeeCourseSessionMedia
                    {
                        CourseSessionMediaId=media.Id,
                        EmployeeCourseSessionId=employeeCourseSessionId,
                        ModifiedDate = DateTime.UtcNow,
                    }
                }
            };

            employeeCourseSessionRepository.Add(employeeCourseSession);
            isNewEmployeeCourseSession = true;
        }
        else
        {
            var employeeCourseSessionMedia = employeeCourseSessionMediaRepository.All
                .FirstOrDefault(d => d.EmployeeCourseSessionId == employeeCourseSession.Id
                    && d.CourseSessionMediaId == media.Id);

            if (employeeCourseSessionMedia == null)
            {
                employeeCourseSessionMediaRepository.Add(new EmployeeCourseSessionMedia
                {
                    EmployeeCourseSessionId = employeeCourseSession.Id,
                    CourseSessionMediaId = media.Id,
                    ModifiedDate = DateTime.UtcNow,
                }); 
            }
            else
            {
                employeeCourseSessionMedia.ModifiedDate = DateTime.UtcNow;
                employeeCourseSessionMediaRepository.Update(employeeCourseSessionMedia);
            }
        }

        // Check if CourseSession has a quiz
        var hasQuiz = await courseSessionQuizRepository.All
            .AnyAsync(q => q.CourseSessionId == media.CourseSessionId, cancellationToken);
        if (!hasQuiz)
        {
            // Get all media IDs for this session
            var allMediaIds = allSessionMedias.Select(m => m.Id).ToList();

            // Get all viewed media IDs for this employee/session
            var viewedMediaIds = await employeeCourseSessionMediaRepository.All
                .Where(x => x.EmployeeCourseSessionId == employeeCourseSession.Id)
                .Select(x => x.CourseSessionMediaId)
                .ToListAsync(cancellationToken);

            viewedMediaIds.Add(request.MediaId); // Include the current media ID

            // If no quiz and all media viewed, mark session as passed and completed
            if (allMediaIds.All(id => viewedMediaIds.Contains(id)))
            {
                employeeCourseSession.IsPassed = true;
                employeeCourseSession.CompletedDate = DateTime.UtcNow;
                if (!isNewEmployeeCourseSession)
                    employeeCourseSessionRepository.Update(employeeCourseSession);
            }
        }

        // Save changes so the new/updated session is persisted
        if (await unitOfWork.SaveAsync(cancellationToken) < 0)
        {
            logger.LogError("error while saving employee course session media tracking");
            return ServiceResponse<bool>.ReturnSuccess();
        }

        // --- NEW LOGIC: Mark EmployeeCourse as Completed if all sessions are passed ---
        var allSessionsPassed = await employeeCourseSessionRepository.All
            .Where(s => s.EmployeeCourseId == employeeCourse.Id)
            .AllAsync(s => s.IsPassed, cancellationToken);

        if (allSessionsPassed)
        {
            employeeCourse.Status = EmployeeCourseStatus.Completed;
            employeeCourse.CompletedDate = DateTime.UtcNow;
        }
        else
        {
            employeeCourse.Status = EmployeeCourseStatus.InProgress;
        }
        employeeCourseRepository.Update(employeeCourse);

        if (await unitOfWork.SaveAsync() < 0)
        {
            logger.LogError("error while saving employee course session media tracking");
        }

        return ServiceResponse<bool>.ReturnSuccess();
    }
}
