﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddNonConformanceCommandHandler(
        INonConformanceRepository _nonConformanceRepository,
        INonConformanceLogRepository _nonConformanceLogRepository,
        UserInfoToken _userInfoToken,
        IUserRepository _userRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddNonConformanceCommandHandler> _logger) : IRequestHandler<AddNonConformanceCommand, ServiceResponse<NonConformanceDto>>
    {
        public async Task<ServiceResponse<NonConformanceDto>> Handle(AddNonConformanceCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var ExistNcNumber = await _nonConformanceRepository.All.Where(c => c.NCNumber == request.NcNumber).FirstOrDefaultAsync();
                if (ExistNcNumber != null)
                {
                    return ServiceResponse<NonConformanceDto>.Return409("This NonConformance Number already exist");
                }
                var existReportedBy = await _userRepository.FindAsync(request.ReportedById);
                if (existReportedBy == null)
                {
                    return ServiceResponse<NonConformanceDto>.Return404("ReportedBy user not found");
                }

                var entity = _mapper.Map<NonConformance>(request);
                entity.Id = Guid.NewGuid();

                var NCLog = _mapper.Map<NonConformanceLog>(entity);
                NCLog.Id = Guid.NewGuid();
                NCLog.CreatedDate = DateTime.UtcNow;
                NCLog.CreatedBy = _userInfoToken.Id;
                NCLog.LogStatus = LogStatus.Created;

                _nonConformanceRepository.Add(entity);
                _nonConformanceLogRepository.Add(NCLog);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<NonConformanceDto>.Return500();
                }

                var entityDto = _mapper.Map<NonConformanceDto>(entity);
                return ServiceResponse<NonConformanceDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving NonConformance");
                return ServiceResponse<NonConformanceDto>.Return500("Error while saving NonConformance");
            }


        }
    }
}
