﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddNonConformanceResponseCommandHandler(
    INonConformanceResponseRepository _nonConformanceResponseRepository,
    INonConformanceRepository _nonConformanceRepository,
    IMapper _mapper,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<AddNonConformanceResponseCommandHandler> _logger,
    UserInfoToken _userInfoToken,
    IUserRepository _userRepository,
    ISendEmailRepository _sendEmailRepository,
    IUserNotificationRepository _userNotificationRepository) : IRequestHandler<AddNonConformanceResponseCommand, ServiceResponse<NonConformanceResponseDto>>
{
    public async Task<ServiceResponse<NonConformanceResponseDto>> Handle(AddNonConformanceResponseCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existNC = await _nonConformanceRepository.FindAsync(request.NonConformanceId);
            if (existNC == null)
            {
                return ServiceResponse<NonConformanceResponseDto>.Return404("NonConformance not Found");
            }
            var entity = _mapper.Map<NonConformanceResponse>(request);
            _nonConformanceResponseRepository.Add(entity);

            // Send Email Responsible Person 
            try
            {
                var userIds = new[] { _userInfoToken.Id, entity.ResponsiblePersonId };
                var users = await _userRepository.All
                    .Where(u => userIds.Contains(u.Id))
                    .ToListAsync();

                var user = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                var responsibleUser = users.FirstOrDefault(u => u.Id == entity.ResponsiblePersonId);
                if (responsibleUser != null)
                {
                    var mainTitle = "Non-Conformance Response Assignment Notification";
                    var subject = "Non-Conformance Response Assignment";
                    var emailMessage = $"You have been assigned as the responsible person for a Non-Conformance Response : <b>{entity.Title}</b>. Please log in to the system to review the details, provide your response, and complete the required actions before the due date.";
                    _sendEmailRepository.AddAsignmentEmails(new SendEmail
                    {
                        Email = responsibleUser.Email,
                        FromName = user.FirstName + ' ' + user.LastName,
                        ToName = responsibleUser.FirstName + ' ' + responsibleUser.LastName,
                        CreatedBy = user.Id,
                        CreatedDate = DateTime.UtcNow,
                    }, emailMessage, mainTitle, subject);

                    // Sent Notification
                    var notificationDto = new UserNotificationDto
                    {
                        Id = entity.Id,
                        UserId = entity.ResponsiblePersonId.Value,
                        NotificationsType = NotificationsType.NC_RESPONSE,
                        Title = entity.Title,
                    };
                    _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                }
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "email does not sent");
            }
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<NonConformanceResponseDto>.Return500();
            }
            var entityDto = _mapper.Map<NonConformanceResponseDto>(entity);
            return ServiceResponse<NonConformanceResponseDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while saving NonConformanceResponse");
            return ServiceResponse<NonConformanceResponseDto>.Return500("Error while saving NonConformanceResponse");
        }
    }
}
