﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class DownloadNonConformanceAttachmentCommandHandler(

    INonConformanceRespAttachmentRepository _nonConformanceRespAttachmentRepository,
     INonConformanceResponseAttachmentChunkRepository _nonConformanceResponseAttachmentChunkRepository,
    IStorageSettingRepository _storageSettingRepository,
    StorageServiceFactory _storageServiceFactory) : IRequestHandler<DownloadNonConformanceAttachmentCommand, ServiceResponse<NonConformanceRespAttachmentDownload>>
{
    public async Task<ServiceResponse<NonConformanceRespAttachmentDownload>> Handle(DownloadNonConformanceAttachmentCommand request, CancellationToken cancellationToken)
    {
        var ncRespAttachment = _nonConformanceRespAttachmentRepository.Find(request.NcRespAttachmentId);
        if (ncRespAttachment == null)
        {
            return ServiceResponse<NonConformanceRespAttachmentDownload>.ReturnFailed(404, "NcAttachment is not found");
        }

        var ncRespAttachmentChunk = await _nonConformanceResponseAttachmentChunkRepository.All
            .Where(c => c.NonConformanceRespAttachmentId == request.NcRespAttachmentId && c.ChunkIndex == request.ChunkIndex)
            .FirstOrDefaultAsync(cancellationToken);

        if (ncRespAttachmentChunk == null)
        {
            return ServiceResponse<NonConformanceRespAttachmentDownload>.ReturnFailed(404, "NcAttachment Chunk is not found");
        }
        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(ncRespAttachment.StorageSettingId);

        if (storeageSetting == null)
        {
            return ServiceResponse<NonConformanceRespAttachmentDownload>.ReturnFailed(404, "Storage setting not found");
        }
        var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
        var fileResult = await storageService.DownloadFileAsync(ncRespAttachmentChunk.Url, storeageSetting.JsonValue, ncRespAttachment.Key, ncRespAttachment.IV);
        if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
        {
            var extension = ncRespAttachmentChunk.Extension;
            if (!extension.StartsWith("."))
                extension = "." + extension;
            NonConformanceRespAttachmentDownload nonConformanceRespAttachmentDownload = new NonConformanceRespAttachmentDownload
            {
                Data = Convert.ToBase64String(fileResult.FileBytes),
                ContentType = FileHelper.GetMimeTypeByExtension(extension),
                FileName = ncRespAttachmentChunk.Url,
                ChunkIndex = ncRespAttachmentChunk.ChunkIndex,
            };
            return ServiceResponse<NonConformanceRespAttachmentDownload>.ReturnResultWith200(nonConformanceRespAttachmentDownload);
        }
        return ServiceResponse<NonConformanceRespAttachmentDownload>.ReturnFailed(500, fileResult.ErrorMessage); ;

    }
}
