﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class MarkAttachmentChunksUploadedCommandHandler(

INonConformanceRespAttachmentRepository nonConformanceRespAttachmentRepository,
INonConformanceResponseAttachmentChunkRepository nonConformanceResponseAttachmentChunkRepository,
IUnitOfWork<QMSDbContext> uow,
IStorageSettingRepository _storageSettingRepository,
StorageServiceFactory _storageServiceFactory,
ILogger<MarkAttachmentChunksUploadedCommandHandler> _logger) : IRequestHandler<MarkAttachmentChunksUploadedCommand, ServiceResponse<NCAttachmentChunkStatus>>
{
    public async Task<ServiceResponse<NCAttachmentChunkStatus>> Handle(MarkAttachmentChunksUploadedCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var nonConformanceRespAttachment = await nonConformanceRespAttachmentRepository.All
                .Where(c => c.Id == request.NonConformanceRespAttachmentId).FirstOrDefaultAsync();

            if (nonConformanceRespAttachment == null)
            {
                return ServiceResponse<NCAttachmentChunkStatus>.ReturnFailed(404, "NCAttachment not found.");
            }

            var NCChunks = nonConformanceResponseAttachmentChunkRepository
                .All
                .Where(c => c.NonConformanceRespAttachmentId == nonConformanceRespAttachment.Id)
                .OrderByDescending(c => c.ChunkIndex)
                .ToList();

            if (NCChunks.Count == 0)
            {
                return ServiceResponse<NCAttachmentChunkStatus>.ReturnFailed(404, " NCAttachment Chunks not found.");
            }


            if (nonConformanceRespAttachment.TotalChunk == NCChunks.Count())
            {
                nonConformanceRespAttachment.IsAllChunkUploaded = request.Status;

                nonConformanceRespAttachmentRepository.Update(nonConformanceRespAttachment);

                if (await uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<NCAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding NCAttachment");
                }
                var response = new NCAttachmentChunkStatus { NonConformanceRespAttachmentId = nonConformanceRespAttachment.Id, Status = request.Status };
                return ServiceResponse<NCAttachmentChunkStatus>.ReturnResultWith200(response);
            }
            else
            {
                if (NCChunks.Count() > 0)
                {
                    foreach (var documentChunk in NCChunks)
                    {
                        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(nonConformanceRespAttachment.StorageSettingId);
                        var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
                        await storageService.DeleteFileAsync(documentChunk.Url, storeageSetting.JsonValue);

                    }
                    nonConformanceResponseAttachmentChunkRepository.RemoveRange(NCChunks);
                }

                nonConformanceRespAttachmentRepository.Remove(nonConformanceRespAttachment);
                if (await uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<NCAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding NCAttachment");
                }

                return ServiceResponse<NCAttachmentChunkStatus>.ReturnFailed(500, "Upload failed: Some parts of the document were not uploaded successfully. The document has been deleted. Please try uploading it again.");
            }
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error while marking NCAttachment chunks uploaded.");
            return ServiceResponse<NCAttachmentChunkStatus>.ReturnFailed(500, "Error while marking NCAttachment chunks uploaded.");
        }
    }
}
