﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateNonConformanceResponseTypeCommandHandler(
        INonConformanceResponseTypeRepository _nonConformanceResponseTypeRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UpdateNonConformanceResponseTypeCommandHandler> _logger) : IRequestHandler<UpdateNonConformanceResponseTypeCommand, ServiceResponse<NonConformanceResponseTypeDto>>
    {
        public async Task<ServiceResponse<NonConformanceResponseTypeDto>> Handle(UpdateNonConformanceResponseTypeCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var existingName = await _nonConformanceResponseTypeRepository
                    .All.Where(n => n.Id != request.Id && n.Name.ToLower() == request.Name.ToLower()).FirstOrDefaultAsync();
                if (existingName != null)
                {
                    return ServiceResponse<NonConformanceResponseTypeDto>.Return409("NonConformanceResponseType name already exist");
                }

                var existingEntity = await _nonConformanceResponseTypeRepository.FindAsync(request.Id);
                if (existingEntity == null)
                {
                    return ServiceResponse<NonConformanceResponseTypeDto>.Return404();
                }

                var entity = _mapper.Map<NonConformanceResponseType>(request);
                _nonConformanceResponseTypeRepository.Update(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<NonConformanceResponseTypeDto>.Return500();
                }
                var entityDto = _mapper.Map<NonConformanceResponseTypeDto>(entity);
                return ServiceResponse<NonConformanceResponseTypeDto>.ReturnResultWith200(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while updating NonConformanceResponseType");
                return ServiceResponse<NonConformanceResponseTypeDto>.Return500("Error while updating NonConformanceResponseType");
            }
        }
    }
}
