﻿
using DocumentFormat.OpenXml.ExtendedProperties;
using MediatR;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.ReminderServices
{
    public class SendEmailSchedulerAssignCourseCommandHandler(
        IEmployeeCourseRepository _employeeCourseRepository,
        IEmailSMTPSettingRepository _emailSMTPSettingRepository, 
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<SendEmailSchedulerAssignCourseCommandHandler> _logger,
        EmailHelper _emailHelper,
        IWebHostEnvironment _webHostEnvironment,
        PathHelper _pathHelper) : IRequestHandler<SendEmailSchedulerAssignCourseCommand, bool>
    {
        public async Task<bool> Handle(SendEmailSchedulerAssignCourseCommand request, CancellationToken cancellationToken)
        {
            var entities = await _employeeCourseRepository.All
                .Where(c => c.EmailStatus == EmailStatus.Pending)
                .Include(c => c.Employee)
                .Include(c => c.Course).ToListAsync();

            var webUrl = _pathHelper.WebUrl;
            if (entities.Count > 0)
            {
                var defaultSmtp = await _emailSMTPSettingRepository.FindBy(c => c.IsDefault).FirstOrDefaultAsync();
                if (defaultSmtp == null)
                {
                    return false;
                }
                
                foreach (var entity in entities)
                {
                    if (!string.IsNullOrEmpty(entity.Employee.Email))
                    {
                        var emailTemplatePath = Path.Combine(_webHostEnvironment.WebRootPath, "EmailTemplates", "assign-course-to-employee-template.html");
                        var emailHtml = System.IO.File.ReadAllText(emailTemplatePath);

                        emailHtml = emailHtml
                            .Replace("{{EmployeeFirstName}}", entity.Employee.FirstName)
                            .Replace("{{EmployeeLastName}}", entity.Employee.LastName)
                            .Replace("{{CourseTitle}}", entity.Course.Title)
                            .Replace("{{SessionLink}}", $"{webUrl}/course/{entity.Link}");

                        try
                        {
                            var email = new SendEmailSpecification
                            {
                                UserName = defaultSmtp.UserName,
                                Password = defaultSmtp.Password,
                                FromAddress = defaultSmtp.FromEmail,
                                ToAddress = entity.Employee.Email,
                                Body = emailHtml,
                                Host = defaultSmtp.Host,
                                Port = defaultSmtp.Port,
                                Subject = "Training Assignment Notification",
                                CCAddress = "",
                                EncryptionType = defaultSmtp.EncryptionType, 
                                FromName = defaultSmtp.FromName
                            };
                            await _emailHelper.SendEmail(email);
                        }
                        catch (Exception ex) 
                        {
                            _logger.LogError(ex.Message, "error while sending Email Specification");
                            return false;
                        }
                    }
                    entity.EmailStatus = EmailStatus.Sent;
                    entity.Course = null;
                    entity.Employee = null;
                }
                _employeeCourseRepository.UpdateRange(entities);
                if (await _uow.SaveAsync() <= 0)
                {
                    return false;
                }
            }
            return true;

        }
    }
}
