﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateReminderCommandHandler(
        IReminderRepository _reminderRepository,
        IReminderUserRepository _reminderUserRepository,
        IDailyReminderRepository _dailyReminderRepository,
        IQuarterlyReminderRepository _quarterlyReminderRepository,
        IHalfYearlyReminderRepository _halfYearlyReminderRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow) : IRequestHandler<UpdateReminderCommand, ServiceResponse<ReminderDto>>
    {
        public async Task<ServiceResponse<ReminderDto>> Handle(UpdateReminderCommand request, CancellationToken cancellationToken)
        {
            var entity = await _reminderRepository
                .AllIncluding(c => c.DailyReminders, cs => cs.QuarterlyReminders, h => h.HalfYearlyReminders, u => u.ReminderUsers).FirstOrDefaultAsync(c => c.Id == request.Id);
            if (entity == null)
            {
                return ServiceResponse<ReminderDto>.Return404();
            }

            if (!request.IsRepeated)
            {
                request.Frequency = Frequency.OneTime;
            }

            _reminderUserRepository.RemoveRange(entity.ReminderUsers);
            _dailyReminderRepository.RemoveRange(entity.DailyReminders);
            _quarterlyReminderRepository.RemoveRange(entity.QuarterlyReminders);
            _halfYearlyReminderRepository.RemoveRange(entity.HalfYearlyReminders);

            if (!request.Frequency.HasValue)
            {
                request.Frequency = Frequency.OneTime;
            }

            var reminder = _mapper.Map(request, entity);
            if (!request.IsRepeated)
            {
                reminder.Frequency = null;
                reminder.DayOfWeek = null;
            }
            _reminderRepository.Update(reminder);

            if (await _uow.SaveAsync() <= -1)
            {
                return ServiceResponse<ReminderDto>.Return500();
            }

            return ServiceResponse<ReminderDto>.ReturnResultWith201(_mapper.Map<ReminderDto>(reminder));
        }
    }
}
