﻿using AutoMapper;
using DocumentFormat.OpenXml.Spreadsheet;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddRiskCommandHandler(
        IRiskRepository _riskRepository,
        IRiskMatrixLevelRepository _riskMatrixLevelRepository,
        IRiskLogRepository _riskLogRepository,
        UserInfoToken _userInfoToken,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddRiskCommandHandler> _logger,
        IUserRepository _userRepository,
        ISendEmailRepository _sendEmailRepository,
        IUserNotificationRepository _userNotificationRepository
        ) : IRequestHandler<AddRiskCommand, ServiceResponse<RiskDto>>
    {
        public async Task<ServiceResponse<RiskDto>> Handle(AddRiskCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entity = _mapper.Map<Risk>(request);
                int riskScore = request.Likelihood * request.Severity;
                var riskLevel = await _riskMatrixLevelRepository
                    .All
                    .Where(c => c.ScoreMin <= riskScore && c.ScoreMax >= riskScore).FirstOrDefaultAsync();
                entity.RiskScore = riskScore;
                entity.RiskLevel = riskLevel.RiskLevel;
                entity.Id = Guid.NewGuid();

                var riskLog = _mapper.Map<RiskLog>(entity);
                riskLog.Id = Guid.NewGuid();
                riskLog.CreatedBy = _userInfoToken.Id;
                riskLog.CreatedDate = DateTime.UtcNow;
                riskLog.LogStatus = LogStatus.Created;

                _riskRepository.Add(entity);
                _riskLogRepository.Add(riskLog);
                // Sent Email asigned User
                try
                {
                    var userIds = new[] { _userInfoToken.Id, entity.AssignedToId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var asignedUser = users.FirstOrDefault(u => u.Id == entity.AssignedToId);
                    if (asignedUser != null)
                    {
                        var mainTitle = "Risk Assignment Notification";
                        var subject = "Risk Assignment";
                        var emailMessage = $"You have been assigned as the responsible person for a Risk: <b>{entity.Title}</b>. Please log in to the system to review the risk details, assess its impact, and take the necessary actions to mitigate or manage it before the due date."; 
                        _sendEmailRepository.AddAsignmentEmails(new SendEmail
                        {
                            Email = asignedUser.Email,
                            FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                            ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                            CreatedBy = fromUser.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, emailMessage, mainTitle, subject);
                        // Sent Notification
                        var notificationDto = new UserNotificationDto
                        {
                            Id = entity.Id,
                            UserId = entity.AssignedToId,
                            NotificationsType = NotificationsType.RISK,
                            Title = entity.Title
                        };
                        _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                    }
                }
                catch (System.Exception ex)
                {
                    _logger.LogError(ex, "email does not sent");
                }
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskDto>.Return500();
                }
                var entityDto = _mapper.Map<RiskDto>(entity);
                return ServiceResponse<RiskDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving risk");
                return ServiceResponse<RiskDto>.Return500("Error while saving risk");
            }
        }
    }
}
