﻿using Amazon.Runtime.Internal.Util;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using Mscc.GenerativeAI;
using QualityManagement.Data;
using QualityManagement.Data.Entities;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadRiskAttachmentChunkCommandHandler(
        IDocumentTokenRepository _documentTokenRepository,
        IRiskAttachmentRepository riskAttachmentRepository,
        IRiskAttachmentChunkRepository riskAttachmentChunkRepository,
        ILogger<DownloadRiskAttachmentChunkCommandHandler> _logger,
        IMediator _mediator) : IRequestHandler<DownloadRiskAttachmentChunkCommand, ServiceResponse<DocumentDownload>>
    {
        public async Task<ServiceResponse<DocumentDownload>> Handle(DownloadRiskAttachmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var isTokenFound = await _documentTokenRepository.All.AnyAsync(c => c.DocumentId == request.Id && c.Token == request.Token);
                if (!isTokenFound)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Token not found");
                }
                var riskAttachment = await riskAttachmentRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (riskAttachment == null)
                {
                    return ServiceResponse<DocumentDownload>.Return404("risk  attachment not found");
                }
                var riskAttachmentChunks = await riskAttachmentChunkRepository.All.Where(c => c.RiskAttachmentId == riskAttachment.Id).OrderBy(c => c.ChunkIndex).ToListAsync();
                if (riskAttachmentChunks.Count <= 0)
                {
                    return ServiceResponse<DocumentDownload>.Return404("risk  attachment chunk not found");
                }
                var chunks = riskAttachmentChunks.Select(c => new FileChunk
                {
                    ChunkIndex = c.ChunkIndex,
                    Url = c.Url,
                }).ToList();
                var fileDownloadRequestCommand = new FileDownloadRequestCommand
                {
                    StorageSettingId = riskAttachment.StorageSettingId,
                    Key = riskAttachment.Key,
                    IV = riskAttachment.IV,
                    Extension = riskAttachment.Extension,
                    Chunks = chunks,
                    FileName = riskAttachment.FileName,
                };
                var result = await _mediator.Send(fileDownloadRequestCommand);
                return result;
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "error while getting risk  attachment");
                return ServiceResponse<DocumentDownload>.Return500("An unexpected fault happened. Try again later. ");
            }
        }
    }
}
