﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class AddRiskReviewCommandHandler(
        IRiskReviewRepository _riskReviewRepository,
        IRiskMatrixLevelRepository _riskMatrixLevelRepository,
        IMapper _mapper,
        ILogger<AddRiskReviewCommandHandler> _logger,
        IUnitOfWork<QMSDbContext> _uow,
        IUserRepository _userRepository,
        UserInfoToken _userInfoToken,
        ISendEmailRepository _sendEmailRepository,
        IUserNotificationRepository _userNotificationRepository,
        IRiskRepository _riskRepository) : IRequestHandler<AddRiskReviewCommand, ServiceResponse<RiskReviewDto>>
    {
        public async Task<ServiceResponse<RiskReviewDto>> Handle(AddRiskReviewCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entity = _mapper.Map<RiskReview>(request);
                int riskScore = request.Likelihood * request.Impact;
                var riskLevel = await _riskMatrixLevelRepository
                    .All
                    .Where(c => c.ScoreMin <= riskScore && c.ScoreMax >= riskScore).FirstOrDefaultAsync();
                entity.RiskScore = riskScore;
                entity.ReviewerId = request.ReviewerId == null ? _userInfoToken.Id : request.ReviewerId.Value;
                entity.RiskLevel = riskLevel.RiskLevel;
                entity.ReviewDate = request.ReviewDate == DateTime.MinValue ? DateTime.UtcNow : request.ReviewDate;
                _riskReviewRepository.Add(entity);
                var risk=await _riskRepository.All.Where(c=>c.Id==request.RiskId).FirstOrDefaultAsync();
                // Send Email Reviewer User 
                try
                {
                    var userIds = new[] { _userInfoToken.Id, entity.ReviewerId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var asignedUser = users.FirstOrDefault(u => u.Id == entity.ReviewerId);
                    if (asignedUser != null)
                    {
                        var mainTitle = "Risk Review Assignment Notification";
                        var subject = "Risk Review Assignment";
                        var emailMessage = $"You have been assigned as the reviewer for a Risk : <b>{risk.Title}</b>. Please log in to the system to review the risk details, evaluate its likelihood and impact, assess the effectiveness of controls, and provide your comments and recommendations before the review date.";
                        _sendEmailRepository.AddAsignmentEmails(new SendEmail
                        {
                            Email = asignedUser.Email,
                            FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                            ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                            CreatedBy = fromUser.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, emailMessage, mainTitle, subject);

                        // Sent Notification
                        var notificationDto = new UserNotificationDto
                        {
                            Id = entity.Id,
                            UserId = entity.ReviewerId,
                            NotificationsType = NotificationsType.RISK_REVIEW,
                            Title = risk != null ? risk.Title : ""
                        }; 
                        _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                    }
                }
                catch (System.Exception ex)
                {
                    _logger.LogError(ex, "email does not sent");
                }
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskReviewDto>.Return500();
                }
                var reviewer = await _userRepository.All.Where(c => c.Id == entity.ReviewerId).FirstOrDefaultAsync();
                if (reviewer != null)
                {
                    entity.Reviewer = reviewer;
                }
                var entityDto = _mapper.Map<RiskReviewDto>(entity);
                return ServiceResponse<RiskReviewDto>.ReturnResultWith201(entityDto);
            }

            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Erorr while saving Riskreview");
                return ServiceResponse<RiskReviewDto>.Return500("Error while saving RiskReview");
            }
        }
    }
}
