﻿using Amazon.Runtime.Internal.Util;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class GetUserNotificationQueryHandler(
    IUserNotificationRepository _userNotificationRepository,
    UserInfoToken _userInfoToken,
    IMapper _mapper,
    ILogger<GetUserNotificationQueryHandler> _logger
    ) : IRequestHandler<GetUserNotificationQuery, List<UserNotificationDto>>
{
    public async Task<List<UserNotificationDto>> Handle(GetUserNotificationQuery request, CancellationToken cancellationToken)
    {
        try
        {
            var userId = _userInfoToken.Id;
            var today = DateTime.UtcNow;
            var fromDate = today.AddDays(-1).AddSeconds(1);
            var toDate = today.AddDays(1).AddSeconds(-1);
            var entities = await _userNotificationRepository
                .All.Include(c => c.Document)
                .Where(c => c.UserId == userId && !c.IsRead
                && (!c.DocumentId.HasValue || !c.Document.IsDeleted)
                && c.CreatedDate > fromDate && c.CreatedDate < toDate)
                .OrderByDescending(c => c.CreatedDate)
                .Take(10)
                 .Select(c => new UserNotificationDto
                 {
                     Id = c.Id,
                     CreatedDate = c.CreatedDate,
                     DocumentId = c.Document != null ? c.DocumentId : null,
                     DocumentName = c.Document != null ? c.Document.Name : "",
                     DocumentNumber = c.Document != null ? c.Document.DocumentNumber : "",
                     IsRead = c.IsRead,
                     Message = c.Message,
                     UserId = c.UserId,
                     Url = c.Document.Url,
                     NotificationsType = c.NotificationsType,
                     JsonPayload = c.JsonPayload
                 })
                .ToListAsync();
            return _mapper.Map<List<UserNotificationDto>>(entities);
        }
        catch(System.Exception ex)
        {
            _logger.LogError(ex, "error while getting notification");
            return new List<UserNotificationDto>();
        }
       
    }
}
