﻿using Amazon.Runtime.Internal.Util;
using AutoMapper;
using MediatR;
using Microsoft.AspNetCore.Identity;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.MediatR.EmployeeCourses;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddUserCommandHandler(
    UserManager<User> _userManager,
    IUserRepository _userRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IMapper _mapper,
    ICourseRepository _courseRepository,
    IEmployeeCourseRepository _employeeCourseRepository,
    ILogger<AddUserCommandHandler> _logger,
     IUserNotificationRepository _userNotificationRepository) : IRequestHandler<AddUserCommand, UserDto>
{
    public async Task<UserDto> Handle(AddUserCommand request, CancellationToken cancellationToken)
    {
        var allUser = await _userRepository.All.IgnoreQueryFilters().ToListAsync();
        var appUser = await _userRepository.All.IgnoreQueryFilters()
            .FirstOrDefaultAsync(c => c.Email.ToLower() == request.Email.ToLower());
        if (appUser != null && appUser.IsDeleted)
        {
            var currentDateTime = $"{DateTime.UtcNow.Year}{DateTime.UtcNow.Month}{DateTime.UtcNow.Day}{DateTime.UtcNow.Hour}{DateTime.UtcNow.Minute}{DateTime.UtcNow.Second}{DateTime.UtcNow.Millisecond}";
            appUser.UserName = $"{appUser.UserName}_IsDeleted_{currentDateTime}";
            var email = appUser.Email.Split('@');

            appUser.Email = $"{email[0]}_IsDeleted_{currentDateTime}{email[1]}";
            appUser.NormalizedUserName = appUser.UserName.ToUpper();
            appUser.NormalizedEmail = appUser.Email.ToUpper();
            _userRepository.Update(appUser);
        }
        if (appUser != null && !appUser.IsDeleted)
        {
            var errorDto = new UserDto
            {
                StatusCode = 409,
                Messages = new List<string> { "Email already exist for another user." }
            };
            return errorDto;
        }
        var entity = _mapper.Map<User>(request);
        entity.Id = Guid.NewGuid();
        entity.ClientId = Guid.NewGuid().ToString();
        entity.ClientSecretHash = ClientSecretGenerator.GenerateClientSecret();

        IdentityResult result = await _userManager.CreateAsync(entity);
        try
        {
            var courses= await _courseRepository.All.Where(c => c.AutoAssignToNewEmployee).ToListAsync();
            foreach (var course in courses)
            {
                var employeeCource = new EmployeeCourse
                {
                    EmployeeId = entity.Id,
                    CourseId = course.Id,
                    Status = EmployeeCourseStatus.Pending,
                    EmailStatus = EmailStatus.Pending,
                    Link = GeneratCourceLink.GenerateLink(Guid.NewGuid().ToString())
                };
                _employeeCourseRepository.Add(employeeCource);
                _userNotificationRepository.AddUserAsignedNewCourseNotification(course.Id, entity.Id, course.Title);
            }
        }
        catch(System.Exception ex)
        {
            _logger.LogError(ex, "Error while asign course to employee");
        }
        
        if (await _uow.SaveAsync() <= -1 && !result.Succeeded)
        {
            var errorDto = new UserDto
            {
                StatusCode = 500,
                Messages = new List<string> { "An unexpected fault happened. Try again later." }
            };
            return errorDto;
        }
        if (!string.IsNullOrEmpty(request.Password))
        {

            string code = await _userManager.GeneratePasswordResetTokenAsync(entity);
            IdentityResult passwordResult = await _userManager.ResetPasswordAsync(entity, code, request.Password);
            if (!passwordResult.Succeeded)
            {
                var errorDto = new UserDto
                {
                    StatusCode = 500,
                    Messages = new List<string> { "An unexpected fault happened. Try again later." }
                };
                return errorDto;
            }
        }
       
        return _mapper.Map<UserDto>(entity);
    }

}
