﻿using AutoMapper;
using MediatR;
using Microsoft.AspNetCore.Identity;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Repository;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class ActiveDeactiveUserCommandHandler(
        UserManager<User> _userManager,
        IMapper _mapper) : IRequestHandler<ActiveDeactiveUserCommand, UserDto>
    {
        public async Task<UserDto> Handle(ActiveDeactiveUserCommand request, CancellationToken cancellationToken)
        {
            var appUser = await _userManager.Users
                .IgnoreQueryFilters()
                .FirstOrDefaultAsync(c => c.Id == request.Id);

            if (appUser == null)
            {
                var errorDto = new UserDto
                {
                    StatusCode = 409,
                    Messages = new List<string> { "User does not exist." }
                };
                return errorDto;
            }
            int userCount = await _userManager.Users.CountAsync();
            if (userCount <= 1)
            {
                var errorDto = new UserDto
                {
                    StatusCode = 409,
                    Messages = new List<string> { "Last remaining user cannot be deleted." }
                };
                return errorDto;
            }

            appUser.IsDeleted = !appUser.IsDeleted;

            IdentityResult result = await _userManager.UpdateAsync(appUser);

            if (!result.Succeeded)
            {
                var errorDto = new UserDto
                {
                    StatusCode = 500,
                    Messages = new List<string> { "An unexpected fault happened. Try again later." }
                };
                return errorDto;
            }

            return _mapper.Map<UserDto>(appUser);
        }
    }
}
