﻿using AutoMapper;
using MediatR;
using Microsoft.AspNetCore.SignalR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateWorkflowCommandHandler(
        IWorkflowRepository _workflowRepository,
        IUnitOfWork<QMSDbContext> _uow,
        IMapper _mapper,
        IHubContext<UserHub, IHubClient> hubContext,
             IConnectionMappingRepository connectionMappingRepository,
             UserInfoToken _userInfo) : IRequestHandler<UpdateWorkflowCommand, ServiceResponse<WorkflowDto>>
    {
        public async Task<ServiceResponse<WorkflowDto>> Handle(UpdateWorkflowCommand request, CancellationToken cancellationToken)
        {
            var entityExist = await _workflowRepository.All
                .FirstOrDefaultAsync(c => c.Name.ToUpper() == request.Name.ToUpper() && c.Id != request.Id);
            if (entityExist != null)
            {
                return ServiceResponse<WorkflowDto>.Return409("Workflow with same name already exists.");
            }
            entityExist = await _workflowRepository.All
                .FirstOrDefaultAsync(c => c.Id == request.Id);
            if (entityExist == null)
            {
                return ServiceResponse<WorkflowDto>.Return409("Workflow does not exists.");
            }
            entityExist.Name = request.Name;
            entityExist.Description = request.Description;
            _workflowRepository.Update(entityExist);

            if (await _uow.SaveAsync() <= 0)
            {
                return ServiceResponse<WorkflowDto>.Return500();
            }
            try
            {
                var user = connectionMappingRepository.GetUserInfoById(_userInfo.Id);

                if (user != null)
                {
                    await hubContext.Clients.AllExcept(new List<string> { user.ConnectionId }).RefreshWorkflowSettings();
                }
            }
            catch (Exception ex)
            {

            }

            var entityDto = _mapper.Map<WorkflowDto>(entityExist);
            return ServiceResponse<WorkflowDto>.ReturnResultWith200(entityDto);
        }
    }
}
