﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;

namespace QualityManagement.Repository;
public class AuditResponseLogList : List<AuditResponseLogDto>
{
    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public AuditResponseLogList(List<AuditResponseLogDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }
    public AuditResponseLogList()
    {

    }

    public async Task<AuditResponseLogList> Create(IQueryable<AuditResponseLog> source, int skip, int pageSize)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize);
        var dtoPageList = new AuditResponseLogList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }

    public async Task<int> GetCount(IQueryable<AuditResponseLog> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<AuditResponseLogDto>> GetDtos(IQueryable<AuditResponseLog> source, int skip, int pageSize)
    {
        try
        {
            var auditTemplates = await source
                .Skip(skip)
                .Take(pageSize)
                .AsNoTracking()
                .Select(c => new AuditResponseLogDto
                {
                    Id = c.Id,
                    AuditNumber = c.Audit.AuditNumber,
                    AuditTemplateId = c.Audit.AuditTemplateId,
                    DepartmentName = c.Audit.AuditTemplate.Department != null ? c.Audit.AuditTemplate.Department.Name : null,
                    Status = c.Status,
                    Title = c.Audit.Title,
                    AuditorName = c.Audit.Auditor != null ? $"{c.Audit.Auditor.FirstName} {c.Audit.Auditor.LastName}" : null,
                    AuditorId = c.Audit.AuditorId,
                    Question = c.Question.Question != null ? c.Question.Question : null,
                    Response = c.Response,
                    CreatedBy = c.CreatedBy,
                    CreatedDate = c.CreatedDate,
                    EffectiveScore = c.Audit.EffectiveScore,
                    TotalScore = c.Audit.TotalScore,
                    LogStatus = c.LogStatus,
                })
                .ToListAsync();
            return auditTemplates;
        }
        catch (Exception ex)
        {
            throw new DataException("Error while getting Audit", ex);
        }
    }
}