﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;

namespace QualityManagement.Repository;

public class CapaRequestLogList : List<CapaRequestLogDto>
{
    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public CapaRequestLogList()
    {

    }
    public CapaRequestLogList(List<CapaRequestLogDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }
    public async Task<CapaRequestLogList> Create(IQueryable<CapaRequestLog> source, int skip, int pageSize)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize);
        var dtoPageList = new CapaRequestLogList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }
    public async Task<int> GetCount(IQueryable<CapaRequestLog> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<CapaRequestLogDto>> GetDtos(IQueryable<CapaRequestLog> source, int skip, int pageSize)
    {
        try
        {
            var capaRequests = await source
                .Skip(skip)
                .Take(pageSize)
                .AsNoTracking()
                .Select(c => new CapaRequestLogDto
                {
                    Id = c.Id,
                    CapaRequestId = c.CapaRequestId,
                    ReferenceNo = c.ReferenceNo,
                    Title = c.Title,
                    Description = c.Description,
                    RaisedById = c.RaisedById,
                    RaisedByName = c.RaisedBY != null ? $"{c.RaisedBY.FirstName} {c.RaisedBY.LastName}" : null,
                    NonConformanceId = c.NonConformanceId.Value,
                    NcNumber = c.NonConformance.NCNumber,
                    AuditId = c.AuditId,
                    AuditName = c.Audit != null ? c.Audit.Title : null,
                    CapaRequestStatus = c.CapaRequestStatus,
                    LogStatus = c.LogStatus,
                    Priority = c.Priority,
                    DueDate = c.DueDate
                })
                .ToListAsync();
            return capaRequests;
        }
        catch (Exception ex)
        {
            throw new DataException("Error while getting CapaRequestsLog", ex);
        }
    }
}
