﻿using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace QualityManagement.Repository
{
    public class ComplaintRepository(IUnitOfWork<QMSDbContext> _uow, IPropertyMappingService _propertyMappingService) : GenericRepository<Complaint, QMSDbContext>(_uow), IComplaintRepository
    {
        public async Task<string> GenerateCompalintNumberAsync()
        {
            int year = DateTime.UtcNow.Year;
            int nextNumber = await All.Where(d => d.ComplaintNo.StartsWith($"{year}-"))
                .CountAsync() + 1;
            return $"{year}-{nextNumber:D5}"; // Example: 2024-00001
        }

        public async Task<ComplaintList> GetComplaint(ComplaintResource complaintResource)
        {
            var collectionBeforePaging = AllIncluding(c => c.AssignedTo,c=>c.ComplaintType);

            collectionBeforePaging = collectionBeforePaging.ApplySort(complaintResource.OrderBy,
           _propertyMappingService.GetPropertyMapping<ComplaintDto, Complaint>());

            if (!string.IsNullOrWhiteSpace(complaintResource.Title))
            {
                collectionBeforePaging = collectionBeforePaging
                .Where(c => c.Title.Contains(complaintResource.Title));
            }
            if (!string.IsNullOrWhiteSpace(complaintResource.ComplaintNo))
            {
                collectionBeforePaging = collectionBeforePaging
                .Where(c => c.ComplaintNo.Contains(complaintResource.ComplaintNo));
            }
            if (!string.IsNullOrWhiteSpace(complaintResource.Description))
            {
                collectionBeforePaging = collectionBeforePaging
                .Where(c => c.Description.Contains(complaintResource.Description));
            }
            if (!string.IsNullOrWhiteSpace(complaintResource.ResolutionSummary))
            {
                collectionBeforePaging = collectionBeforePaging
                .Where(c => c.ResolutionSummary.Contains(complaintResource.ResolutionSummary));
            }
            if (complaintResource.AssignedToId.HasValue)
            {
                collectionBeforePaging = collectionBeforePaging
              .Where(c => c.AssignedToId == complaintResource.AssignedToId);
            }
            if (complaintResource.ComplaintTypeId.HasValue)
            {
                collectionBeforePaging = collectionBeforePaging
              .Where(c => c.ComplaintTypeId == complaintResource.ComplaintTypeId);
            }
            if (complaintResource.Priority.HasValue)
            {
                collectionBeforePaging = collectionBeforePaging
              .Where(c => c.Priority == complaintResource.Priority);
            }
            if (complaintResource.ReportedBy.HasValue)
            {
                collectionBeforePaging = collectionBeforePaging
              .Where(c => c.ReportedBy == complaintResource.ReportedBy);
            }
            if (complaintResource.Status.HasValue)
            {
                collectionBeforePaging = collectionBeforePaging
              .Where(c => c.Status == complaintResource.Status);
            }
            if (complaintResource.ComplaintStartDate.HasValue && complaintResource.ComplaintEndDate.HasValue)
            {
                var startDate = complaintResource.ComplaintStartDate.Value;
                var endDate = complaintResource.ComplaintEndDate.Value;

                DateTime minDate = new DateTime(startDate.Year, startDate.Month, startDate.Day, 0, 0, 0);
                DateTime maxDate = new DateTime(endDate.Year, endDate.Month, endDate.Day, 23, 59, 59);

                collectionBeforePaging = collectionBeforePaging
                            .Where(c => c.ComplaintDate >= minDate &&
                                c.ComplaintDate <= maxDate);
            }
            var complaintList = new ComplaintList();
            return await complaintList.Create(
                collectionBeforePaging,
                complaintResource.Skip,
                complaintResource.PageSize);

        }
    }
}
