﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic.Core;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data;
using QualityManagement.Data.Dto;

namespace QualityManagement.Repository;
public class EmployeeCourseList : List<EmployeeDto>
{
    public EmployeeCourseList()
    {
    }

    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public EmployeeCourseList(List<EmployeeDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }

    public async Task<EmployeeCourseList> Create(IQueryable<User> source, int skip, int pageSize, List<Guid> courseEmployeeIds, string orderBy)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize, courseEmployeeIds, orderBy);
        var dtoPageList = new EmployeeCourseList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }

    public async Task<int> GetCount(IQueryable<User> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<EmployeeDto>> GetDtos(IQueryable<User> source, int skip, int pageSize, List<Guid> courseEmployeeIds, string orderBy)
    {
        var entities = await source
            .Select(c => new EmployeeDto
            {
                Id = c.Id,
                Email = c.Email,
                FirstName = c.FirstName,
                LastName = c.LastName,
                IsAssigned = courseEmployeeIds.Contains(c.Id)
            })
            .OrderByDescending(c => c.IsAssigned)
            .ThenBy(orderBy)
            .Skip(skip)
            .Take(pageSize)
            .AsNoTracking()
            .ToListAsync();
        return entities;
    }
}