﻿using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace QualityManagement.Repository;
public class RecentDocumentRepository(
    IPropertyMappingService _propertyMappingService,
    IUnitOfWork<QMSDbContext> uow,
    UserInfoToken _userInfoToken) : GenericRepository<DocumentAuditTrail, QMSDbContext>(uow), IRecentDocumentRepository
{
    public async Task<RecentDocumentList> GetRecentDocuments(DocumentResource documentResource)
    {
        var documentOperations = new[]
        {
            DocumentOperation.Read,DocumentOperation.Created,DocumentOperation.Modified,
            DocumentOperation.Archived,DocumentOperation.Restored,DocumentOperation.Download,
            DocumentOperation.Added_Version,DocumentOperation.Restored_Version,DocumentOperation.Download,
            DocumentOperation.Added_Signature,DocumentOperation.Deleted,DocumentOperation.Added_Folder,
            DocumentOperation.Edited_Folder,DocumentOperation.Archived_Folder,DocumentOperation.Restored_Folder,
            DocumentOperation.Deleted_Folder
        };

        var collectionBeforePaging = AllIncluding(c => c.CreatedByUser, d => d.Document, c => c.Document.Category, c => c.Category)
            .Where(c => documentOperations.Contains(c.OperationName) && c.CreatedBy == _userInfoToken.Id);

        collectionBeforePaging =
            collectionBeforePaging
            .IgnoreQueryFilters()
            .ApplySort(documentResource.OrderBy,
            _propertyMappingService.GetPropertyMapping<DocumentAuditTrailDto, DocumentAuditTrail>());
        if (!string.IsNullOrWhiteSpace(documentResource.Name))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => EF.Functions.Like(c.Document.Name, $"%{documentResource.Name}%"));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.DocumentNumber))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => EF.Functions.Like(c.Document.DocumentNumber, $"{documentResource.DocumentNumber}%"));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.Id))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.DocumentId == Guid.Parse(documentResource.Id));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.CategoryId))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.CategoryId == Guid.Parse(documentResource.CategoryId));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.Operation))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.OperationName == Enum.Parse<DocumentOperation>(documentResource.Operation));
        }
        var recentDocumentList = new RecentDocumentList();
        return await recentDocumentList.Create(
            collectionBeforePaging,
            documentResource.Skip,
            documentResource.PageSize
        );
    }
}
