﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;

namespace QualityManagement.Repository.RiskManagment;

public class RiskList : List<RiskDto>
{
    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public RiskList()
    {

    }
    public RiskList(List<RiskDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }
    public async Task<RiskList> Create(IQueryable<Risk> source, int skip, int pageSize)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize);
        var dtoPageList = new RiskList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }
    public async Task<int> GetCount(IQueryable<Risk> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<RiskDto>> GetDtos(IQueryable<Risk> source, int skip, int pageSize)
    {
        try
        {
            var risk = await source
                .Skip(skip)
                .Take(pageSize)
                .AsNoTracking()
                .Select(c => new RiskDto
                {
                    Id = c.Id,
                    Title = c.Title,
                    Description = c.Description,
                    Status = c.Status,
                    Severity = c.Severity,
                    Likelihood = c.Likelihood,
                    RiskCategoryId = c.RiskCategoryId,
                    RiskScore = c.RiskScore,
                    RiskLevel = c.RiskLevel,
                    AssignedToId = c.AssignedToId,
                    CreatedBy = c.CreatedBy,
                    CreatedDate = c.CreatedDate,
                    RiskCategoryName = c.RiskCategory != null ? c.RiskCategory.Name : "",
                    AssignedToName = c.AssignedTo != null ? $"{c.AssignedTo.FirstName} {c.AssignedTo.LastName}" : "",
                    CreatedByName = c.CreatedUser != null ? $"{c.CreatedUser.FirstName} {c.CreatedUser.LastName}" : ""
                })
                .ToListAsync();
            return risk;
        }
        catch (Exception ex)
        {
            throw new DataException("Error while getting Risk", ex);
        }
    }

}
