﻿using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace QualityManagement.Repository.RiskManagment
{
    public class RiskLogList : List<RiskLogDto>
    {
        public int Skip { get; private set; }
        public int TotalPages { get; private set; }
        public int PageSize { get; private set; }
        public int TotalCount { get; private set; }

        public RiskLogList()
        {

        }
        public RiskLogList(List<RiskLogDto> items, int count, int skip, int pageSize)
        {
            TotalCount = count;
            PageSize = pageSize;
            Skip = skip;
            TotalPages = (int)Math.Ceiling(count / (double)pageSize);
            AddRange(items);
        }
        public async Task<RiskLogList> Create(IQueryable<RiskLog> source, int skip, int pageSize)
        {
            var count = await GetCount(source);
            var dtoList = await GetDtos(source, skip, pageSize);
            var dtoPageList = new RiskLogList(dtoList, count, skip, pageSize);
            return dtoPageList;
        }
        public async Task<int> GetCount(IQueryable<RiskLog> source)
        {
            return await source.AsNoTracking().CountAsync();
        }
        public async Task<List<RiskLogDto>> GetDtos(IQueryable<RiskLog> source, int skip, int pageSize)
        {
            try
            {
                var riskLog = await source
                    .Skip(skip)
                    .Take(pageSize)
                    .AsNoTracking()
                    .Select(c => new RiskLogDto
                    {
                        Id = c.Id,
                        RiskId = c.RiskId,
                        LogStatus = c.LogStatus,
                        Title = c.Title,
                        Description = c.Description,
                        Status = c.Status,
                        Severity = c.Severity,
                        Likelihood = c.Likelihood,
                        RiskCategoryId = c.RiskCategoryId,
                        RiskScore = c.RiskScore,
                        RiskLevel = c.RiskLevel,
                        AssignedToId = c.AssignedToId,
                        CreatedBy = c.CreatedBy,
                        CreatedDate = c.CreatedDate,
                        RiskCategoryName = c.RiskCategory != null ? c.RiskCategory.Name : "",
                        AssignedToName = c.AssignedTo != null ? $"{c.AssignedTo.FirstName} {c.AssignedTo.LastName}" : "",
                        CreatedByName = c.CreatedByUser != null ? $"{c.CreatedByUser.FirstName} {c.CreatedByUser.LastName}" : ""
                    })
                    .ToListAsync();
                return riskLog;
            }
            catch (Exception ex)
            {
                throw new DataException("Error while getting Risk Log", ex);
            }
        }
    }
}