﻿using System;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;

namespace QualityManagement.Repository;
public class SupplierRepository(IUnitOfWork<QMSDbContext> uow, IPropertyMappingService _propertyMappingService) : GenericRepository<Supplier, QMSDbContext>(uow),
          ISupplierRepository
{
    public async Task<string> GenerateSupplierNumberAsync()
    {
        int year = DateTime.UtcNow.Year;
        int nextNumber = await All.Where(d => d.SupplierCode.StartsWith($"SC-{year}-"))
            .CountAsync() + 1;
        return $"SC-{year}-{nextNumber:D4}"; // Example: SC-2024-00001
    }

    public async Task<SupplierList> GetSuppliers(SupplierResource supplierResource)
    {

        var collectionBeforePaging = All;

        collectionBeforePaging = collectionBeforePaging.ApplySort(supplierResource.OrderBy,
            _propertyMappingService.GetPropertyMapping<SupplierDto, Supplier>());

        if (!string.IsNullOrWhiteSpace(supplierResource.SupplierCode))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.SupplierCode.Contains(supplierResource.SupplierCode));
        }
        if (!string.IsNullOrWhiteSpace(supplierResource.Name))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.Name.Contains(supplierResource.Name));
        }
        if (!string.IsNullOrWhiteSpace(supplierResource.ContactPerson))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.ContactPerson.Contains(supplierResource.ContactPerson));
        }
        if (!string.IsNullOrWhiteSpace(supplierResource.Email))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.Email.Contains(supplierResource.Email));
        }
        if (!string.IsNullOrWhiteSpace(supplierResource.Phone))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.Phone.Contains(supplierResource.Phone));
        }
        if (supplierResource.Status.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.Status == supplierResource.Status);
        }

        var supplierList = new SupplierList();
        return await supplierList.Create(
            collectionBeforePaging,
            supplierResource.Skip,
            supplierResource.PageSize);
    }
}
