﻿using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens.Jwt;
using System.Linq;
using System.Security.Claims;
using System.Text;
using System.Threading.Tasks;
using System.Web;
using Microsoft.EntityFrameworkCore;
using Microsoft.IdentityModel.Tokens;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;

namespace QualityManagement.Repository;

public class UserRepository(
   IUnitOfWork<QMSDbContext> uow,
   ICompanyProfileRepository _companyProfileRepository,
    IUserClaimRepository _userClaimRepository,
    IRoleClaimRepository _roleClaimRepository,
    IUserRoleRepository _userRoleRepository,
    IPropertyMappingService _propertyMappingService,
     IEmployeeCourseRepository _employeeCourseRepository,
     IActionRepository _actionRepository,
     JwtSettings _settings = null) : GenericRepository<User, QMSDbContext>(uow),
      IUserRepository
{
    public async Task<List<User>> GetUsersByIds(List<Guid> ids)
    {
        return await All.Where(cs => ids.Contains(cs.Id)).Distinct().ToListAsync();
    }
    public async Task<List<Guid>> GetUsersByRoleId(Guid roleId)
    {
        return await All.Where(cs => cs.UserRoles.Any(c => c.RoleId == roleId)).Select(c => c.Id).ToListAsync();
    }

    private async Task<List<string>> GetRoleClaims(User appUser)
    {
        if (appUser.IsSuperAdmin)
        {
            var roleClaims = await _actionRepository.All.Select(c => c.Code).ToListAsync();
            return roleClaims;
        }
        else
        {
            var rolesIds = await _userRoleRepository.All.Where(c => c.UserId == appUser.Id)
                      .Select(c => c.RoleId)
                      .ToListAsync();

            var roleClaims = await _roleClaimRepository.All.Where(c => rolesIds.Contains(c.RoleId)).Select(c => c.ClaimType).ToListAsync();
            return roleClaims;
        }

    }
    private async Task<List<string>> GetUserAndRoleClaims(User appUser)
    {
        var userClaims = await _userClaimRepository.FindBy(c => c.UserId == appUser.Id).Select(c => c.ClaimType).ToListAsync();
        var roleClaims = await GetRoleClaims(appUser);
        var finalClaims = userClaims;
        finalClaims.AddRange(roleClaims);
        finalClaims = finalClaims.Distinct().ToList();
        return finalClaims;
    }

    public async Task<UserAuthDto> BuildUserAuthObject(User appUser)
    {
        var companyProfile = _companyProfileRepository.All.FirstOrDefault();
        UserAuthDto ret = new UserAuthDto();
        List<AppClaimDto> appClaims = new List<AppClaimDto>();
        // Set User Properties
        ret.Id = appUser.Id.ToString();
        ret.UserName = appUser.UserName;
        ret.FirstName = appUser.FirstName;
        ret.LastName = appUser.LastName;
        ret.Email = appUser.Email;
        ret.PhoneNumber = appUser.PhoneNumber;
        ret.IsSuperAdmin = appUser.IsSuperAdmin;
        ret.IsAuthenticated = true;
        // Get all claims for this user
        var appClaimDtos = await GetUserAndRoleClaims(appUser);
        ret.Claims = appClaimDtos.Select(c => c).ToList();
        var claims = appClaimDtos.Select(c => new Claim(c, "true")).ToList(); // Convert to List<Claim>
        claims.Add(new Claim("licensekey", string.IsNullOrEmpty(companyProfile.LicenseKey) ? "" : HttpUtility.UrlEncode(companyProfile.LicenseKey.ToString())));
        claims.Add(new Claim("purchasecode", string.IsNullOrEmpty(companyProfile.PurchaseCode) ? "" : HttpUtility.UrlEncode(companyProfile.PurchaseCode.ToString())));

        // Set JWT bearer token
        ret.BearerToken = BuildJwtToken(ret, claims, appUser.Id);
        return ret;
    }
    protected string BuildJwtToken(UserAuthDto authUser, IList<Claim> claims, Guid Id)
    {
        SymmetricSecurityKey key = new SymmetricSecurityKey(
          Encoding.UTF8.GetBytes(_settings.Key));

        // Create the signing credentials using the key and algorithm
        var credentials = new SigningCredentials(key, SecurityAlgorithms.HmacSha256);

        claims.Add(new Claim(Microsoft.IdentityModel.JsonWebTokens.JwtRegisteredClaimNames.Sub.ToString(), Id.ToString()));
        claims.Add(new Claim("Email", authUser.Email));
        claims.Add(new Claim("IsSuperAdmin", authUser.IsSuperAdmin.ToString()));
        // Create the JwtSecurityToken object
        var tokenDescriptor = new JwtSecurityToken(
          issuer: _settings.Issuer,
          audience: _settings.Audience,
          claims: claims,
          notBefore: DateTime.UtcNow,
          expires: DateTime.UtcNow.AddMinutes(
              _settings.MinutesToExpiration),
          signingCredentials: credentials
        );
        var tokenHandler = new JwtSecurityTokenHandler();
        // Create a string representation of the Jwt token
        return tokenHandler.WriteToken(tokenDescriptor);
    }

    public async Task<UserList> GetUsers(UserResource userResource)
    {
        var collectionBeforePaging = All;
        collectionBeforePaging =
           collectionBeforePaging.ApplySort(userResource.OrderBy,
           _propertyMappingService.GetPropertyMapping<UserDto, User>());

        if (!string.IsNullOrWhiteSpace(userResource.SearchQuery))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => EF.Functions.Like(c.FirstName, $"%{userResource.SearchQuery}%")
                || EF.Functions.Like(c.LastName, $"%{userResource.SearchQuery}%")
                || EF.Functions.Like(c.Email, $"%{userResource.SearchQuery}%"));
        }

        var users = new UserList();

        return await users.Create(
            collectionBeforePaging,
            userResource.Skip,
            userResource.PageSize
            );
    }

    public async Task<EmployeeCourseList> GetCourseEmployees(EmployeeResource employeeResource)
    {
        var collectionBeforePaging = All;
        //collectionBeforePaging =
        //   collectionBeforePaging.ApplySort(employeeResource.OrderBy,
        //   _propertyMappingService.GetPropertyMapping<EmployeeDto, Employee>());

        var employeeCourseCollection = _employeeCourseRepository.AllIncluding(c => c.Employee);

        if (!string.IsNullOrWhiteSpace(employeeResource.FirstName))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c =>
               EF.Functions.Like(c.FirstName, $"%{employeeResource.FirstName}%"));

            employeeCourseCollection
                .Where(c => EF.Functions.Like(c.Employee.FirstName, $"%{employeeResource.FirstName}%"));
        }

        if (!string.IsNullOrWhiteSpace(employeeResource.LastName))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c =>
               EF.Functions.Like(c.LastName, $"%{employeeResource.LastName}%"));

            employeeCourseCollection = employeeCourseCollection
                .Where(c => EF.Functions.Like(c.Employee.LastName, $"%{employeeResource.LastName}%"));
        }

        if (!string.IsNullOrWhiteSpace(employeeResource.Email))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c =>
               EF.Functions.Like(c.Email, $"%{employeeResource.Email}%"));

            employeeCourseCollection = employeeCourseCollection
                .Where(c => EF.Functions.Like(c.Employee.Email, $"%{employeeResource.Email}%"));
        }

        if (employeeResource.CourseId.HasValue)
        {
            employeeCourseCollection = employeeCourseCollection
                .Where(c => c.CourseId == employeeResource.CourseId);
        }

        var courseEmployeeIds = await employeeCourseCollection.Select(c => c.EmployeeId).ToListAsync();

        var employees = new EmployeeCourseList();

        return await employees.Create(collectionBeforePaging,
            employeeResource.Skip,
            employeeResource.PageSize, courseEmployeeIds, employeeResource.OrderBy ?? "FirstName");
    }
}
