import { inject, Injectable } from '@angular/core';
import { catchError, Observable, of } from 'rxjs';
import { AuditTemplate } from './models/audit-template';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { HttpClient } from '@angular/common/http';
import { CommonError } from '@core/error-handler/common-error';

@Injectable({
  providedIn: 'root'
})
export class AuditTemplateService {

  private commonHttpErrorService = inject(CommonHttpErrorService);
  private httpClient = inject(HttpClient);

  public icons: string[] = [
    'trending_up',      // activity
    'airplay',          // airplay
    'error_outline',    // alert-circle
    'report_problem',   // alert-octagon
    'warning',          // alert-triangle
    'archive',          // archive
    'arrow_downward',   // arrow-down
    'arrow_back',       // arrow-left
    'arrow_forward',    // arrow-right
    'arrow_upward',     // arrow-up
    'emoji_events',     // award
    'bar_chart',        // bar-chart
    'battery_std',      // battery
    'notifications',    // bell
    'menu_book',        // book
    'bookmark',         // bookmark
    'calendar_today',   // calendar
    'photo_camera',     // camera
    'check',            // check
    'check_circle',     // check-circle
    'expand_more',      // chevron-down
    'chevron_left',     // chevron-left
    'chevron_right',    // chevron-right
    'expand_less',      // chevron-up
    'content_paste',    // clipboard
    'schedule',         // clock
    'cloud',            // cloud
    'code',             // code
    'explore',          // compass
    'content_copy',     // copy
    'credit_card',      // credit-card
    'storage',          // database
    'file_download',    // download
    'edit',             // edit
    'visibility',       // eye
    'insert_drive_file',// file
    'movie',            // film
    'folder',           // folder
    'public',           // globe
    'grid_view',        // grid
    'favorite',         // heart
    'home',             // home
    'image',            // image
    'info',             // info
    'inbox',            // inbox
    'layers',           // layers
    'link',             // link
    'list',             // list
    'lock',             // lock
    'login',            // log-in
    'logout',           // log-out
    'mail',             // mail
    'map',              // map
    'menu',             // menu
    'chat',             // message-circle
    'mic',              // mic
    'remove',           // minus
    'dark_mode',        // moon
    'more_horiz',       // more-horizontal
    'more_vert',        // more-vertical
    'inventory_2',      // package
    'attach_file',      // paperclip
    'pause',            // pause
    'phone',            // phone
    'play_arrow',       // play
    'add',              // plus
    'print',            // printer
    'refresh',          // refresh-ccw
    'repeat',           // repeat
    'fast_rewind',      // rewind
    'save',             // save
    'search',           // search
    'send',             // send
    'settings',         // settings
    'share',            // share
    'shield',           // shield
    'shopping_cart',    // shopping-cart
    'tune',             // sliders
    'star',             // star
    'light_mode',       // sun
    'label',            // tag
    'delete',           // trash
    'trending_up',      // trending-up
    'file_upload',      // upload
    'person',           // user
    'verified_user',    // user-check
    'person_add',       // user-plus
    'groups',           // users
    'videocam',         // video
    'volume_up',        // volume-2
    'wifi',             // wifi
    'close',            // x
    'flash_on'          // zap
  ];

  getAllTemplates(): Observable<AuditTemplate[]> {
    const url = 'auditTemplate';
    return this.httpClient.get<AuditTemplate[]>(url);

  }

  getTemplateById(id: string): Observable<AuditTemplate> {
    const url = 'auditTemplate/' + id;
    return this.httpClient.get<AuditTemplate>(url);
  }

  getQuestionsForTemplate(id: string): Observable<any[] | CommonError> {
    const url = `auditTemplate/${id}/questions`;
    return this.httpClient.get<any[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  createTemplate(template: AuditTemplate): Observable<AuditTemplate | CommonError> {
    const url = `auditTemplate`;
    return this.httpClient.post<AuditTemplate>(url, template)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateTemplate(id: string, template: AuditTemplate): Observable<AuditTemplate | CommonError> {
    const url = `auditTemplate/` + id;
    return this.httpClient.put<AuditTemplate>(url, template)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteTemplate(id: string): Observable<boolean | CommonError> {
    const url = `department/${id}`;
    return this.httpClient.delete<boolean>(url).pipe(
      catchError(this.commonHttpErrorService.handleError)
    );
  }
}
