import {
  CdkDragDrop,
  DragDropModule,
  moveItemInArray,
} from '@angular/cdk/drag-drop';
import { Component, inject, OnInit } from '@angular/core';
import {
  AbstractControl,
  FormArray,
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatFormFieldModule } from '@angular/material/form-field';
import { MatIconModule } from '@angular/material/icon';
import { MatInputModule } from '@angular/material/input';
import { MatSelectModule } from '@angular/material/select';
import { MatSlideToggleModule } from '@angular/material/slide-toggle';
import { MatTooltipModule } from '@angular/material/tooltip';
import { ActivatedRoute, Router, RouterModule } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { QuestionFormComponent } from './question-form/question-form.component';
import { QUESTION_TYPE } from './models/question-type';
import { AuditTemplate } from './models/audit-template';
import { AuditTemplateService } from './audit-template-service';
import { AUDIT_TEMPLATE_SECTION } from './models/audit-template-section';
import { AUDIT_TYPE } from './models/audit-type';
import { compareOptionValidator } from './validator/compare-option-validator';
import { DepartmentStore } from './department/store/department.store';
import { BaseComponent } from '../base.component';
import { Department } from '@core/domain-classes/department';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { NgClass } from '@angular/common';
import { MatDialog } from '@angular/material/dialog';
import { ManageDepartmentComponent } from './department/manage-department/manage-department.component';
import { ToastrService } from '@core/services/toastr-service';

@Component({
  selector: 'app-audit-template',
  imports: [
    TranslateModule,
    RouterModule,
    ReactiveFormsModule,
    MatButtonModule,
    MatIconModule,
    MatFormFieldModule,
    MatInputModule,
    MatSelectModule,
    MatTooltipModule,
    MatCardModule,
    DragDropModule,
    QuestionFormComponent,
    MatSlideToggleModule,
    HasClaimDirective,
    NgClass
  ],
  templateUrl: './audit-template.component.html',
  styleUrl: './audit-template.component.scss',
})
export class AuditTemplateComponent extends BaseComponent implements OnInit {
  templateForm!: FormGroup<any>;
  isEditMode = false;
  saving = false;
  templateId: string | null = null;
  expandedQuestionIndex: number | null = null;
  questionTypes = Object.keys(QUESTION_TYPE)
    .filter((key) => !isNaN(Number(QUESTION_TYPE[key as any]))) // filter numeric values
    .map((key) => ({
      label: key,
      value: QUESTION_TYPE[key as keyof typeof QUESTION_TYPE],
    })) ?? [];

  sections = Object.keys(AUDIT_TEMPLATE_SECTION)
    .filter((key) => !isNaN(Number(AUDIT_TEMPLATE_SECTION[key as any]))) // filter numeric values
    .map((key) => ({
      label: key,
      value: AUDIT_TEMPLATE_SECTION[key as keyof typeof AUDIT_TEMPLATE_SECTION],
    }));

  auditTypes = Object.keys(AUDIT_TYPE)
    .filter((key) => !isNaN(Number(AUDIT_TYPE[key as any]))) // filter numeric values
    .map((key) => ({
      label: key,
      value: AUDIT_TYPE[key as keyof typeof AUDIT_TYPE],
    }));

  departments: Department[] = [];
  icons: string[] = [];
  departmentStore = inject(DepartmentStore);
  dialog = inject(MatDialog);
  toaster = inject(ToastrService);

  constructor(
    private fb: FormBuilder,
    private route: ActivatedRoute,
    private router: Router,
    private templateService: AuditTemplateService,
    private commonDialogService: CommonDialogService,
  ) {
    super();
  }

  ngOnInit(): void {
    this.icons = this.templateService.icons; // Load icons from the service
    this.getDepartments();
    this.initForm();
    this.sub$.sink = this.route.paramMap.subscribe((params) => {
      const id = params.get('id');
      if (id != 'add') {
        this.templateId = id;
        this.isEditMode = !!this.templateId;
        if (this.isEditMode && this.templateId) {
          this.loadTemplate(this.templateId);
        }
      }
    });
  }

  initForm(): void {
    this.templateForm = this.fb.group({
      name: ['', [Validators.required, Validators.maxLength(100)]],
      description: ['', [Validators.maxLength(500)]],
      isActive: [true],
      departmentId: [null, [Validators.required]],
      auditType: [null, [Validators.required]],
      questions: this.fb.array([]),
      iconName: [null, [Validators.required]],
    });
  }

  loadTemplate(id: string): void {
    this.templateService.getTemplateById(id).subscribe({
      next: (template: AuditTemplate) => {
        if (template) {
          this.populateForm(template);
        } else {
          this.toaster.error(this.translationService.getValue('TEMPLATE_NOT_FOUND'));

          this.router.navigate(['/audit-templates']);
        }
      },
      error: (error) => {
        this.toaster.error(this.translationService.getValue('ERROR_LOADING_TEMPLATE'));
      }
    });
  }

  populateForm(template: AuditTemplate): void {
    this.templateForm.patchValue({
      name: template.name,
      description: template.description,
      isActive: template.isActive,
      auditType: template.auditType,
      departmentId: template.departmentId,
      iconName: template.iconName,
    });
    // Clear existing questions
    this.questions.clear();

    // Add each question
    if (template.questions && template.questions.length > 0) {
      template.questions.forEach((question) => {
        const questionGroup = this.fb.group({
          id: [question.id],
          question: [
            question.question,
            [Validators.required, Validators.maxLength(500)],
          ],
          description: [question.description, [Validators.maxLength(1000)]],
          inputType: [question.inputType, [Validators.required]],
          section: [question.section],
          orderNo: [question.orderNo],
          requiredEvidence: [question.requiredEvidence],
          questionOptions: this.fb.array([]),
          maxScore: [question.maxScore, [Validators.required, Validators.pattern('^[1-9][0-9]*$')]],
        });
        // Add options if present
        const optionsArray = questionGroup.get('questionOptions') as FormArray;
        if (question.questionOptions && question.questionOptions.length > 0) {
          question.questionOptions.forEach((option) => {
            if (question.inputType === QUESTION_TYPE.RATING) {
              optionsArray.push(
                this.fb.group(
                  {
                    id: [option.id],
                    optionText: [
                      option.optionText,
                      [Validators.required, Validators.min(1)],
                    ],
                    optionValue: [option.optionValue, [Validators.required]],
                    orderNo: [option.orderNo],
                  },
                  { validators: compareOptionValidator() }
                )
              );
            }
            if (
              question.inputType === QUESTION_TYPE.MULTI_SELECT ||
              question.inputType === QUESTION_TYPE.SINGLE_SELECT
            ) {
              optionsArray.push(
                this.fb.group({
                  id: [option.id],
                  optionText: [
                    option.optionText,
                    [Validators.required, Validators.maxLength(50)],
                  ],
                  optionValue: [option.optionValue, [Validators.required]],
                  orderNo: [option.orderNo],
                })
              );
            }
          });
        }
        this.questions.push(questionGroup);
      });
    }
  }

  get questions(): FormArray {
    return this.templateForm.get('questions') as FormArray;
  }

  addQuestion(): void {
    const questionGroup = this.fb.group({
      question: ['', [Validators.required, Validators.maxLength(500)]],
      description: ['', [Validators.maxLength(1000)]],
      inputType: [QUESTION_TYPE.TEXT, [Validators.required]],
      isRequired: [false],
      section: [null, [Validators.required]],
      requiredEvidence: [false],
      orderNo: [this.questions.length + 1],
      questionOptions: this.fb.array([]),
      maxScore: [null, [Validators.required, Validators.pattern('^[1-9][0-9]*$'), Validators.max(100), Validators.min(1)]],
    });

    this.questions.push(questionGroup);
    // Expand the newly added question
    this.expandedQuestionIndex = this.questions.length - 1;

    // Scroll to the new question
    setTimeout(() => {
      const element = document.getElementById(
        `question-${this.expandedQuestionIndex}`
      );
      if (element) {
        element.scrollIntoView({ behavior: 'smooth', block: 'center' });
      }
    }, 100);
  }

  getDepartments(): void {
    this.departmentStore.loadDepartments();
  }

  removeQuestion(index: number): void {
    this.sub$.sink = this.commonDialogService
      .deleteConfirmtionDialog(
        this.translationService.getValue(
          'ARE_YOU_SURE_YOU_WANT_TO_REMOVE_THIS_QUESTION'
        )
      )
      .subscribe((isConfirmed) => {
        if (isConfirmed) {
          this.questions.removeAt(index);
          // Update the order for remaining questions
          this.updateQuestionOrder();
        }
      });
  }

  updateQuestionOrder(): void {
    this.questions.controls.forEach(
      (control: AbstractControl, index: number) => {
        control.get('orderNo')?.setValue(index + 1);
      }
    );
  }

  onQuestionDrop(event: CdkDragDrop<string[]>): void {
    moveItemInArray(
      this.questions.controls,
      event.previousIndex,
      event.currentIndex
    );
    this.updateQuestionOrder();
  }

  toggleQuestion(index: number): void {
    this.expandedQuestionIndex =
      this.expandedQuestionIndex === index ? null : index;
  }

  onSubmit(): void {
    if (this.templateForm.invalid) {
      this.templateForm.markAllAsTouched();
      this.toaster.error(this.translationService.getValue('PLEASE_FIX_THE_FORM_ERRORS_BEFORE_SUBMITTING'));
      return;
    }

    this.saving = true;
    const formValue = this.templateForm.value;

    const template: AuditTemplate = {
      ...formValue,
      id: this.isEditMode ? this.templateId! : undefined,
    };

    const action = this.isEditMode
      ? this.templateService.updateTemplate(this.templateId ?? '', template)
      : this.templateService.createTemplate(template);

    this.sub$.sink = action.subscribe({
      next: (result: any) => {
        this.saving = false;
        const message = this.isEditMode
          ? this.translationService.getValue('TEMPLATE_UPDATED_SUCCESSFULLY')
          : this.translationService.getValue('TEMPLATE_CREATED_SUCCESSFULLY');
        this.toaster.success(message);
        this.router.navigate(['/audit/audit-template']);
      },
      error: (error: any) => {
        this.saving = false;
      },
    });
  }

  addDepartment(): void {
    const dialogRef = this.dialog.open(ManageDepartmentComponent, {
      width: '500px',
    });
    dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        this.templateForm.get('departmentId')?.setValue(result.id);
      }
    });
  }
}
