import { Component, inject, OnInit } from '@angular/core';
import { MatTableModule } from '@angular/material/table';
import { Department } from '@core/domain-classes/department';
import { BaseComponent } from '../../../base.component';
import { debounceTime, distinctUntilChanged } from 'rxjs';
import { DepartmentStore } from '../store/department.store';
import { MatDialog } from '@angular/material/dialog';
import { ManageDepartmentComponent } from '../manage-department/manage-department.component';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { MatMenuModule } from '@angular/material/menu';
import { MatIconModule } from '@angular/material/icon';
import { FormControl, ReactiveFormsModule } from '@angular/forms';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-department-list',
  imports: [
    MatTableModule,
    MatMenuModule,
    MatIconModule,
    ReactiveFormsModule,
    HasClaimDirective,
    PageHelpTextComponent,
    TranslateModule,
    MatButtonModule,
    MatCardModule,
    NgClass
  ],
  templateUrl: './department-list.component.html',
  styleUrl: './department-list.component.scss'
})
export class DepartmentListComponent extends BaseComponent implements OnInit {
  private dialog = inject(MatDialog);
  commandDialogService = inject(CommonDialogService);
  displayedColumns: string[] = ['action', 'departmentName', 'description'];
  displayedColumns1: string[] = ['action-search', 'departmentName-search', 'description-search']
  departmentStore = inject(DepartmentStore);
  filterDepartmentName: FormControl = new FormControl();

  ngOnInit(): void {
    this.departmentNameChangeValue();
  }

  departmentNameChangeValue() {
    this.filterDepartmentName.valueChanges.pipe(
      debounceTime(500),
      distinctUntilChanged()
    ).subscribe((value: string) => {
      this.departmentStore.setDepartmentNameFilter(value);
    });
  }

  addDepartment() {
    this.dialog.open(ManageDepartmentComponent, {
      width: '500px',
    })
  }

  updateDepartment(department: Department) {
    this.dialog.open(ManageDepartmentComponent, {
      width: '500px',
      data: Object.assign({}, department)
    })
  }

  deleteDepartment(departmentId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_DEPARMENT")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.departmentStore.deleteDepartmentById(departmentId);
        }
      }
    });
  }

  refresh() {
    this.departmentStore.loadDepartments();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.departmentStore.departments().indexOf(row);
  }
}
