import { computed, inject } from "@angular/core";
import { Department } from "@core/domain-classes/department";
import { CommonError } from "@core/error-handler/common-error";
import { patchState, signalStore, withComputed, withHooks, withMethods, withState } from "@ngrx/signals";
import { DepartmentService } from "../department.service";
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from "@core/services/translation.service";
import { rxMethod } from "@ngrx/signals/rxjs-interop";
import { debounceTime, distinctUntilChanged, pipe, switchMap, tap } from "rxjs";
import { tapResponse } from "@ngrx/operators";
import { toObservable } from "@angular/core/rxjs-interop";

type DepartmentState = {
  searchQuery: string;
  departments: Department[];
  department: Department | null;
  loadList: boolean;
  isAddUpdate: boolean;
};

export const initialDepartmentState: DepartmentState = {
  searchQuery: '',
  departments: [],
  department: null,
  loadList: true,
  isAddUpdate: false,
};

export const DepartmentStore = signalStore(
  { providedIn: 'root' },
  withState(initialDepartmentState),
  withComputed(({ departments, searchQuery }) => ({
    filteredDepartments: computed(() => {
      const filter = searchQuery().toLowerCase();
      return departments()
        .filter(dept => dept?.name?.toLowerCase().includes(filter));
    })
  })),
  withMethods(
    (
      store,
      departmentService = inject(DepartmentService),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService),
    ) => ({
      setDepartmentNameFilter(Value: string) {
        patchState(store, { searchQuery: Value });
      },
      loadDepartments: rxMethod<void>(
        pipe(
          debounceTime(300),
          tap(() => patchState(store, { departments: [] })),
          switchMap(() =>
            departmentService.getDepartments().pipe(
              tapResponse({
                next: (departments: Department[]) => {
                  patchState(store, {
                    departments: [...departments],
                    loadList: false,
                  });
                },
                error: (err) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      addDepartment: rxMethod<Department>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((department: Department) =>
            departmentService.addDepartment(department).pipe(
              tapResponse({
                next: (response) => {
                  const newDepartment: Department = response as Department;
                  toastrService.success(
                    translationService.getValue('DEPARTMENT_CREATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    loadList: true,
                    isAddUpdate: true,
                    department: newDepartment
                  });
                },
                error: (err: CommonError) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      updateDepartment: rxMethod<Department>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((department: Department) =>
            departmentService.updateDepartment(department).pipe(
              tapResponse({
                next: (response) => {
                  const updateDepartment: Department = response as Department;
                  toastrService.success(
                    translationService.getValue('DEPARTMENT_UPDATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    departments: store.departments().map((dep) => {
                      return updateDepartment.id === dep.id ? updateDepartment : dep;
                    }),
                    department: updateDepartment,
                    isAddUpdate: true
                  });
                },
                error: (err: CommonError) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      deleteDepartmentById: rxMethod<string>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((departmentId: string) =>
            departmentService.deleteDepartment(departmentId).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('DEPARTMENT_DELETED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    departments: store.departments().filter((w) => w.id !== departmentId),
                  });
                },
                error: (err: CommonError) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false })
      },
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadDepartments();
        }
      });
    },
  })
);
